import re
import sform

from bson.objectid import ObjectId
from collections import defaultdict
from datetime import date
from itertools import chain, groupby
from operator import itemgetter
from typing import List, AnyStr, Dict, Optional, Any, Set, Iterable

from django.db.models import Q
from django.conf import settings
from django.core.exceptions import ValidationError

from staff.departments.controllers.department import DepartmentCtl
from staff.departments.controllers.proposal import (
    ProposalCtl,
    get_mongo_objects,
    from_mongo_id,
)
from staff.departments.controllers.proposal_action import (
    DEPARTMENT_ACTION_SECTIONS,
    is_creating,
    is_deleting,
    is_moving,
)
from staff.departments.models import Department, ProposalMetadata, DepartmentKind, DEPARTMENT_CATEGORY
from staff.departments.edit.constants import HR_CONTROLLED_DEPARTMENT_ROOTS
from staff.lib.utils.qs_values import localize
from staff.lib.forms.validators import latin_field_validator
from staff.lib.models.departments_chain import get_departments_tree
from staff.lib.models.mptt import get_descendants_query
from staff.person.models import Staff

from staff.proposal.forms.base import (
    ProposalBaseForm,
    ProposalEntityActionForm,
    GridFieldWithMeta,
    readonly_if_locked,
    actions_matcher,
)


class DepartmentName(ProposalBaseForm):
    name = sform.CharField(max_length=255, default='', state=sform.REQUIRED)
    name_en = sform.CharField(max_length=255, default='', state=sform.REQUIRED)

    hr_type = sform.BooleanField(default=True)
    is_correction = sform.BooleanField(default=False, state=sform.NORMAL)

    @staticmethod
    def clean_name(value):
        return clean_name(value)

    @staticmethod
    def clean_name_en(english_value):
        latin_field_validator(english_value)
        return clean_name(english_value)

    def clean_is_correction(self, value):
        author_user = self.base_initial.get('author_user')
        if value and author_user and not author_user.has_perm('django_intranet_stuff.can_execute_department_proposals'):
            raise ValidationError(
                'No permissions for field `is_correction`',
                code='no_permission',
            )
        return value

    def structure_as_dict(self, prefix=''):
        result = super(DepartmentName, self).structure_as_dict(prefix)
        author_user = self.base_initial.get('author_user')
        user_has_no_permission = (
             author_user
             and not author_user.has_perm('django_intranet_stuff.can_execute_department_proposals')
        )

        if user_has_no_permission:
            result.pop('is_correction', None)

        return result


class DepartmentHierarchy(ProposalBaseForm):
    parent = sform.SuggestField(
        queryset=Department.objects.filter(intranet_status=1),
        to_field_name='url',
        label_fields='name',
    )
    fake_parent = sform.CharField(max_length=32)
    changing_duties = sform.NullBooleanField()  # При перемещении подразделения - обязательное

    def get_field_state(self, name):
        if name == 'changing_duties':
            is_moving = not self.base_initial.get('creating_department')
            if is_moving:
                return sform.REQUIRED

        return super(DepartmentHierarchy, self).get_field_state(name)

    def clean(self):
        if not self.cleaned_data['parent'] and not self.cleaned_data['fake_parent']:
            self._errors[('parent', )] = [
                ValidationError(
                    'Parent of fake_parent must be filled',
                    code='required',
                )
            ]
        return self.cleaned_data

    @staticmethod
    def clean_parent(parent_department):
        return parent_department and parent_department.url


class DepartmentAdministration(ProposalBaseForm):
    chief = sform.SuggestField(
        queryset=Staff.objects.filter(is_dismissed=False),
        to_field_name='login',
        label_fields=('first_name', 'last_name', ),
    )
    deputies = sform.MultipleSuggestField(
        queryset=Staff.objects.filter(is_dismissed=False),
        to_field_name='login',
        label_fields={
            'caption': ('first_name', 'last_name'),
            'extra_fields': ['login'],
        },
    )

    def clean(self):
        cleaned_data = self.cleaned_data
        single_person_fields = ('chief', 'budget_holder')
        multiple_person_fields = ('deputies', 'hr_partners')
        for field in single_person_fields:
            if cleaned_data.get(field):
                cleaned_data[field] = cleaned_data[field].login

        for field in multiple_person_fields:
            if field in cleaned_data:
                cleaned_data[field] = [
                    person.login for person in cleaned_data[field]
                ]

        return cleaned_data


class DepartmentTechnicalFields(ProposalBaseForm):
    code = sform.CharField(max_length=255, state=sform.REQUIRED)
    department_type = sform.ModelChoiceField(
        queryset=(
            DepartmentKind.objects
            .filter(intranet_status=1)
            .order_by('name')
        ),
        label_extractor='name',
        state=sform.REQUIRED,
        empty_label='-'
    )
    category = sform.ChoiceField(
        choices=DEPARTMENT_CATEGORY.choices(
            include_empty=True,
            empty_label='-',
        ),
        state=sform.REQUIRED,
    )
    order = sform.IntegerField()
    allowed_overdraft_percents = sform.DecimalField(
        state=sform.NORMAL,
        min_value=0.0,
        max_value=1000.0,
        decimal_places=2,
    )

    @readonly_if_locked
    def get_field_state(self, name):
        if name == 'code':
            if not self.base_initial.get('creating_department'):
                return sform.READONLY

        return super(DepartmentTechnicalFields, self).get_field_state(name)

    @staticmethod
    def clean_code(value):
        value = value.strip().lower()
        if not re.match(r'^[0-9a-zA-Z]{1,20}$', value):
            raise ValidationError(
                'Code value does not validate',
                code='invalid'
            )
        return value

    def clean_allowed_overdraft_percents(self, value):
        author_user = self.base_initial['author_user']
        if value and not author_user.has_perm('django_intranet_stuff.can_execute_department_proposals'):
            raise ValidationError(
                'No permissions for field `allowed_overdraft_percents`',
                code='no_permission',
            )
        return value

    @staticmethod
    def clean_department_type(kind_obj):
        return kind_obj and int(kind_obj.id)

    @staticmethod
    def clean_order(order_value):
        return order_value or 0

    def structure_as_dict(self, prefix=''):
        result = super().structure_as_dict(prefix)
        author_user = self.base_initial.get('author_user')
        user_has_no_permission = (
            author_user
            and not author_user.has_perm('django_intranet_stuff.can_execute_department_proposals')
        )

        if user_has_no_permission:
            result.pop('allowed_overdraft_percents', None)

        return result

    def clean(self):
        cleaned_data = self.cleaned_data
        for decimal_field in ('allowed_overdraft_percents',):
            value = cleaned_data.get(decimal_field)
            cleaned_data[decimal_field] = value if value is None else str(value)

        return cleaned_data


class DepartmentEditForm(ProposalEntityActionForm):
    fake_id = sform.CharField(max_length=17)
    url = sform.CharField(max_length=120)

    sections = sform.MultipleChoiceField(choices=DEPARTMENT_ACTION_SECTIONS)
    name = sform.FieldsetField(DepartmentName, trim_on_empty=True)
    hierarchy = sform.FieldsetField(DepartmentHierarchy, trim_on_empty=True)
    administration = sform.FieldsetField(DepartmentAdministration, trim_on_empty=True)
    technical = sform.FieldsetField(DepartmentTechnicalFields, trim_on_empty=True)

    delete = sform.BooleanField()

    def __init__(self, data=None, initial=None, *args, **kwargs):
        super(DepartmentEditForm, self).__init__(data, initial, *args, **kwargs)

        self.base_initial['creating_department'] = True
        if (data and data.get('url')) or (initial and initial.get('url')):
            self.base_initial['creating_department'] = False

    def get_field_state(self, name):
        if name in ('name', 'technical', 'hierarchy'):  # fieldset`ы, содержащие обязательные поля
            if self.base_initial.get('creating_department'):
                return sform.REQUIRED
        # todo: Сделать sections обязательными если не выбрано delete
        return super(DepartmentEditForm, self).get_field_state(name)

    def clean_url(self, value):
        if not value:
            return ''

        return value

    def check_renaming(self, dep_url):
        actual_names = (
            Department.objects
            .values('name', 'name_en')
            .get(url=dep_url)
        )
        new_names = {
            key: value
            for key, value in self.cleaned_data['name'].items()
            if key in ('name', 'name_en')
        }
        error_code = 'name_changes_required'

        if new_names == actual_names:
            self._errors[('name', )] = [
                ValidationError(
                    'Names are not changed',
                    code=error_code,
                )
            ]
            return

        if new_names['name'] == actual_names['name']:
            user_have_correction_permission = (
                self
                .base_initial['author_user']
                .has_perm('django_intranet_stuff.can_execute_department_proposals')
            )
            is_correction = self.cleaned_data['name'].get('is_correction', False)

            if user_have_correction_permission:
                error_code = 'is_correction_required'
                if is_correction:
                    return

            self._errors[('name', )] = [
                ValidationError(
                    'For changing name_en fill correction flag',
                    code=error_code,
                )
            ]

    def check_not_empty(self, cleaned_data: List[Dict]) -> None:
        if not any([
            cleaned_data.get(n) for n, v in self.base_fields.items()
            if isinstance(v, sform.FieldsetField) or n == 'delete'
        ]):
            raise ValidationError(
                'Department changes cannot be empty',
                code='cannot_be_empty',
            )

    def check_url_or_fake_id_exists(self, cleaned_data: List[Dict], dep_url: AnyStr) -> None:
        if not (dep_url or cleaned_data.get('fake_id')):
            self._errors[('url', )] = [
                ValidationError(
                    'Either url or fake_id value should be provided',
                    code='configuration_error'
                )
            ]

    @staticmethod
    def _get_conflict_dep(name: AnyStr, name_en: AnyStr, exclude_dep_url: bool) -> Optional[Dict[AnyStr, AnyStr]]:
        conflict_dep = (
            Department.objects
            .filter(Q(name__iexact=name) | Q(name_en__iexact=name_en))
            .filter(intranet_status=1)
            .exclude(url=exclude_dep_url)
            .values('url', 'name', 'name_en')
        )

        return conflict_dep.first()

    @staticmethod
    def _get_conflict_proposal(name, name_en, is_hr_controlled, exclude_proposal_id):
        #  type (AnyStr, AnyStr, bool) -> Optional[Dict[AnyStr, AnyStr]]

        if is_hr_controlled:
            root_filter = {'root_departments': {'$in': list(HR_CONTROLLED_DEPARTMENT_ROOTS.keys())}}
        else:
            root_filter = {'root_departments': {'$nin': list(HR_CONTROLLED_DEPARTMENT_ROOTS.keys())}}

        unfinished_proposal_ids = list(
            ProposalMetadata.objects
            .filter(applied_at=None, deleted_at=None)
            .exclude(proposal_id=exclude_proposal_id)
            .values_list('proposal_id', flat=True)
        )

        mongo_spec = {
            '_id': {'$in': [ObjectId(_id) for _id in unfinished_proposal_ids]},
            'actions': {'$ne': []},
            'actions.name': {'$exists': True},
        }
        mongo_spec.update(root_filter)
        mongo_objects = get_mongo_objects(mongo_spec)
        for mongo_object in mongo_objects:
            for dep_action in mongo_object['actions']:
                if (
                    'name' in dep_action
                    and (
                        dep_action['name']['name'].lower() == name.lower()
                        or dep_action['name']['name_en'].lower() == name_en.lower()
                    )
                ):
                    return {
                        '_id': str(mongo_object['_id']),
                        'author': mongo_object['author'],
                        'name': dep_action['name']['name'],
                        'name_en': dep_action['name']['name_en'],
                    }
        return None

    def check_name_is_unique(self, cleaned_data, exclude_dep_url=None, exclude_proposal_id=None):
        # type: (List[Dict], Optional[AnyStr], Optional[AnyStr]) -> None
        if 'name' not in cleaned_data:
            return
        hr_controlled = self.base_initial['changes_controlled_by_hr']
        new_name = cleaned_data['name']['name']
        new_name_en = cleaned_data['name']['name_en']

        conflict_dep_data = self._get_conflict_dep(new_name, new_name_en, exclude_dep_url)
        if conflict_dep_data:
            collision_field = 'name' if conflict_dep_data['name'].lower() == new_name.lower() else 'name_en'
            self._errors[('name', collision_field)] = [
                ValidationError(
                    '{} should be unique within HR controlled departments'.format(collision_field),
                    code='name_conflict_with_department',
                    params={'conflict_with_department': conflict_dep_data['url']},
                )
            ]

        conflict_proposal_data = self._get_conflict_proposal(new_name, new_name_en, hr_controlled, exclude_proposal_id)
        if conflict_proposal_data:
            collision_field = 'name' if conflict_proposal_data['name'].lower() == new_name.lower() else 'name_en'
            author_data = (
                Staff.objects
                .filter(id=conflict_proposal_data['author'])
                .values('login', 'first_name', 'last_name', 'first_name_en', 'last_name_en')
                .first()
            ) or dict.fromkeys(['login', 'first_name', 'last_name', 'first_name_en', 'last_name_en'], '')
            localize(author_data)

            self._errors[('name', collision_field)] = [
                ValidationError(
                    '{} should be unique but there is another proposal with same value'.format(collision_field),
                    code='name_conflict_with_proposal',
                    params={
                        'conflict_with_proposal': conflict_proposal_data['_id'],
                        'conflict_proposal_author_login': author_data['login'],
                        'conflict_proposal_author_first_name': author_data['first_name'],
                        'conflict_proposal_author_last_name': author_data['last_name'],
                    }
                )
            ]

    def check_conflict_proposals(self, dep_url):  # type: (AnyStr) -> None
        this_proposal_id = self.base_initial.get('_id', '')
        conflict_proposals = find_conflict_proposals(
            department_url=dep_url,
            exclude=this_proposal_id,
        )

        if conflict_proposals:
            self._errors[('url', )] = [
                ValidationError(
                    'There is unfinished proposal with current department',
                    code='department_conflict',
                    params={'proposal_uid': conflict_proposals[0]},
                )
            ]

    def clean(self):
        this_proposal_id = self.base_initial.get('_id')
        cleaned_data = self.cleaned_data
        self.check_not_empty(cleaned_data)

        dep_url = cleaned_data.get('url')
        self.check_url_or_fake_id_exists(cleaned_data, dep_url)

        if not dep_url:  # creating department
            self.check_name_is_unique(cleaned_data, exclude_proposal_id=this_proposal_id)
            return cleaned_data

        if 'name' in self.cleaned_data:
            self.check_renaming(dep_url)
            self.check_name_is_unique(cleaned_data, exclude_dep_url=dep_url, exclude_proposal_id=this_proposal_id)

        self.check_conflict_proposals(dep_url)

        existing_department = (
            Department.objects
            .filter(url=dep_url, intranet_status=1)
            .first()
        )

        if not existing_department:
            self._errors[('url', )] = [
                ValidationError(
                    f'Department with url {dep_url} does not exist',
                    code='department_does_not_exist',
                    params={'dep_url': dep_url},
                )
            ]
        elif cleaned_data.get('delete', False):
            ctl = DepartmentCtl(existing_department)

            if ctl.has_population():
                self._errors[('url', )] = [
                    ValidationError(
                        f'Department with url {dep_url} is not empty',
                        code='department_not_empty',
                        params={'dep_url': dep_url},
                    )
                ]

        return cleaned_data


class DepartmentGridFieldWithMeta(GridFieldWithMeta):
    def get_tree_id(self, action_values: List) -> int or None:
        if not action_values:
            return None

        urls = (
            [av['url'] for av in action_values if av['url']]
            or [av['hierarchy']['parent'] for av in action_values if av['hierarchy']['parent']]
        )

        if not urls:
            return None

        return (
            Department.objects
            .filter(url=urls[0])
            .values_list('tree_id', flat=True)
            .first()
        )

    def clean(self, new_value, old_value, required, trim_on_empty, base_initial, base_data):
        base_initial['changes_controlled_by_hr'] = (
            self.get_tree_id(action_values=new_value) in HR_CONTROLLED_DEPARTMENT_ROOTS.values()
        )
        return super(DepartmentGridFieldWithMeta, self).clean(
            new_value,
            old_value,
            required,
            trim_on_empty,
            base_initial,
            base_data,
        )

    def populate_with_meta(self, fields_list, base_initial: Dict[str, Any]):
        from staff.proposal.controllers import DepartmentDataCtl
        dep_urls = [
            action['value']['url']['value']
            for action in fields_list['value']
        ]
        dep_data_ctl = DepartmentDataCtl([url for url in dep_urls if url], base_initial['author_user'])

        for field_dict in fields_list['value']:
            if field_dict['value']['url']['value']:
                field_dict['meta'] = dep_data_ctl.as_meta(field_dict['value']['url']['value'])
            else:
                field_dict['meta'] = {}


class DepartmentChangesProposalForm(ProposalBaseForm):

    actions = DepartmentGridFieldWithMeta(
        sform.FieldsetField(DepartmentEditForm),
        values_matcher=actions_matcher,
    )

    def __init__(self, *args, **kwargs):
        super(DepartmentChangesProposalForm, self).__init__(*args, **kwargs)

    def clean_actions(self, actions):
        """
        Проверка на циклы в создаваемых и перемещаемых подразделениях
        Проверка на вхождение одного подразделения в заявку больше одного раза
        Проверка на недублирование названий в рамках одной заявки
        Проверка на перемещение под удаляемое подразделение в рамках одной заявки
        """

        duplicate_urls = self._get_duplicate_urls(actions)
        for url, action_numbers in duplicate_urls.items():
            for action_number in action_numbers:
                self._errors[('actions', action_number, 'url', 'url')] = [
                    ValidationError(
                        'Duplicate department on {} positions in actions'.format(str(duplicate_urls[url])),
                        code='department_conflict',
                    )
                ]

        validator = DepartmentConsistancyValidator(actions)
        validator.check_moving_to_deleting_department()
        validator.check_moving_to_deleting_department_another_proposal()
        validator.check_creating_cycles()

        code_conflicting_actions = validator.find_creating_dep_with_duplicate_code()
        for action_number, code_value in code_conflicting_actions:
            self._errors[('actions', action_number, 'technical', 'code')] = [
                ValidationError(
                    'Department with code {} already exist here'.format(code_value),
                    code='department_code_conflict',
                    params={}
                )
            ]

        duplicate_name_indexes = self._get_duplicate_name_actions(actions)

        if duplicate_name_indexes['name']:
            for action_numbers in duplicate_name_indexes['name']:
                for action_number in action_numbers:
                    self._errors[('actions', action_number, 'name', 'name')] = [
                        ValidationError(
                            'Duplicate department_name on {} positions in actions'.format(str(action_numbers)),
                            code='name_conflict_with_same_proposal',
                        )
                    ]

        if duplicate_name_indexes['name_en']:
            for action_numbers in duplicate_name_indexes['name_en']:
                for action_number in action_numbers:
                    self._errors[('actions', action_number, 'name', 'name_en')] = [
                        ValidationError(
                            'Duplicate department_name on {} positions in actions'.format(str(action_numbers)),
                            code='name_conflict_with_same_proposal',
                        )
                    ]

        return actions

    @staticmethod
    def _get_duplicate_urls(actions):
        seen_urls = {}

        for index, action in enumerate(actions):
            if action['url']:
                seen_urls.setdefault(action['url'], []).append(index)

        return {key: value for key, value in seen_urls.items() if len(value) > 1}

    @staticmethod
    def clean_apply_at(value):
        if value < date.today():
            raise ValidationError(
                'Apply date should be at least today',
                code='invalid',
            )
        return value

    @staticmethod
    def _get_duplicate_name_actions(actions):  # type: (List[Dict]) -> Dict[AnyStr, List[List]]
        """Собираем списки индексов экшенов, где name или name_en одинаковы"""
        ru_names, en_names = defaultdict(list), defaultdict(list)

        for index, action in enumerate(actions):
            if 'name' not in action:
                continue
            ru_name_value = clean_name(action['name']['name']).lower()
            en_name_value = clean_name(action['name']['name_en']).lower()
            ru_names[ru_name_value].append(index)
            en_names[en_name_value].append(index)

        return {
            'name': [indexes for name, indexes in ru_names.items() if len(indexes) > 1],
            'name_en': [indexes for name, indexes in en_names.items() if len(indexes) > 1],
        }


def _proposals_with_ticket(ticket_key):
    proposals_with_this_ticket = list(get_mongo_objects(
        spec={
            '$or': [
                # Так храним в старой заявке
                {'tickets.department_ticket': ticket_key},
                {'tickets.department_linked_ticket': ticket_key},
            ],
        }
    ))
    for p in proposals_with_this_ticket:
        yield from_mongo_id(p['_id'])


class DepartmentConsistancyValidator:
    def __init__(self, actions):
        self.actions = actions
        self._deleting_urls = None

        self.urls = [act['url'] for act in self.actions if act['url']]
        self.urls.extend(
            [
                act['hierarchy']['parent']
                for act in self.actions
                if 'hierarchy' in act and act['hierarchy']['parent']
            ]
        )
        self.urls = list(set(self.urls))
        self.parents = self._get_parents_dict()

    @property
    def deleting_urls(self) -> Set[str]:
        if self._deleting_urls is None:
            deleting_urls = {act['url'] for act in filter(is_deleting, self.actions)}
            moving_urls = {act['url'] for act in filter(is_moving, self.actions)}
            self._deleting_urls = (
                self._get_all_descendant_urls(deleting_urls) -
                self._get_all_descendant_urls(moving_urls)
            )
        return self._deleting_urls

    def _get_parents_dict(self):
        id2url = dict(
            Department.objects
            .filter(url__in=self.urls)
            .values_list('id', 'url')
        )
        chains = get_departments_tree(
            departments=id2url.keys(),
            fields=['url', 'parent__url'],
        )
        self.check_controlled_is_not_mixed_with_uncontrolled(chains)
        parents = {}
        for dep_id, dep_chain in chains.items():
            for dep_data in dep_chain:
                parents[dep_data['url']] = dep_data['parent__url']

        # parents сформирован по реальным данным как {dep: parent}
        # теперь дополним его данными из заявки
        for action in self.actions:
            if action['fake_id']:
                parent = action['hierarchy']['parent'] or action['hierarchy']['fake_parent']
                parents[action['fake_id']] = parent
            elif 'hierarchy' in action:
                parents[action['url']] = action['hierarchy']['parent'] or action['hierarchy']['fake_parent']

        return parents

    @staticmethod
    def _get_all_descendant_urls(department_urls: Iterable[str]) -> Set[str]:
        departments = Department.objects.filter(url__in=department_urls).values('tree_id', 'lft', 'rght')
        return set(
            Department.objects
            .filter(intranet_status=1)
            .filter(get_descendants_query(departments, include_self=True))
            .values_list('url', flat=True)
        )

    def check_controlled_is_not_mixed_with_uncontrolled(self, departments_chains: Dict[int, List]) -> None:
        top_departments = {ch[0]['url'] for ch in departments_chains.values()}
        if top_departments - set(HR_CONTROLLED_DEPARTMENT_ROOTS.keys()) in (set(), top_departments):
            return
        raise ValidationError(
            'Mixed departments from controlled and uncontrolled branches',
            code='mixed_department_branches',
        )

    def check_moving_to_deleting_department_another_proposal(self):
        unfinished_proposal_ids = list(
            ProposalMetadata.objects
            .filter(applied_at=None, deleted_at=None)
            .values_list('proposal_id', flat=True)
        )
        mongo_spec = {
            'actions.delete': True,
            '_id': {'$in': [ObjectId(_id) for _id in unfinished_proposal_ids]}
        }
        actions = list(chain.from_iterable([doc['actions'] for doc in get_mongo_objects(spec=mongo_spec, limit=999)]))
        actions_with_id = [act for act in actions if act.get('id')]

        id_to_url = dict(
            Department.objects
            .filter(id__in=[act['id'] for act in actions_with_id])
            .values_list('id', 'url')
        )
        deleting_urls = {id_to_url[act['id']] for act in filter(is_deleting, actions_with_id)}
        moving_urls = {id_to_url[act['id']] for act in filter(is_moving, actions_with_id)}
        deleting_urls = (
            self._get_all_descendant_urls(deleting_urls) -
            self._get_all_descendant_urls(moving_urls)
        )
        self.check_moving_to_deleting_department(deleting_urls)

    def check_moving_to_deleting_department(self, deleting_urls: Optional[Set[str]] = None):
        deleting_urls = deleting_urls or self.deleting_urls
        for action in self.actions:
            if 'hierarchy' in action and action['hierarchy']['parent']:
                if action['hierarchy']['parent'] in deleting_urls:
                    raise ValidationError(
                        'Moving to deleting department detected',
                        code='wrong_relations',
                        params={
                            'department': action['url'] or action['fake_id'],
                            'deleting_parent': action['hierarchy']['parent'],
                        }
                    )

    def check_creating_cycles(self):
        def trace_to_root(dep_url, trace=None):
            trace = trace or [dep_url]
            if self.parents[dep_url] is None:
                return trace
            if self.parents[dep_url] not in trace:
                trace.append(self.parents[dep_url])
                return trace_to_root(self.parents[dep_url], trace)
            else:
                raise ValidationError(
                    'Cyclic department relations detected',
                    code='cyclic_relations',
                    params={
                        'departments': trace,
                    }
                )
        for action in self.actions:
            trace_to_root(action['url'] or action['fake_id'])

    def find_creating_dep_with_duplicate_code(self):
        child_codes_qs = (
            Department.objects
            .filter(parent__url__in=self.parents.keys())
            .values_list('parent__url', 'code')
            .order_by('parent__url')
        )
        child_codes = defaultdict(list)
        for parent_url, group in groupby(child_codes_qs, itemgetter(0)):
            for p, code in group:
                child_codes[parent_url].append(code)

        codes_conflicts = []
        for num, action in enumerate(filter(is_creating, self.actions)):
            parent_url = action['hierarchy']['parent']
            new_department_code = action['technical']['code']
            if parent_url and new_department_code in child_codes[parent_url]:
                codes_conflicts.append((num, new_department_code))
        return codes_conflicts


def find_conflict_proposals(department_url, exclude=None):
    """Возвращает список _id всех невыполненных заявок, содержащих указанное подразделение"""
    try:
        department_id = (
            Department.objects
            .filter(url=department_url)
            .values_list('id', flat=True)
            .get()
        )
    except (Department.DoesNotExist, Department.MultipleObjectsReturned):
        return []

    proposal_filter = {'actions.id': department_id}
    if exclude:
        proposal_filter['_id'] = {'$ne': ObjectId(exclude)}

    another_proposals_with_url = [
        ctl.proposal_id for ctl in ProposalCtl.filter(proposal_filter)
    ]
    return list(
        ProposalMetadata.objects
        .filter(
            proposal_id__in=another_proposals_with_url,
            applied_at=None,
            deleted_at=None,
        )
        .values_list('proposal_id', flat=True)
    )


def normalize_link(link):
    if link.startswith(settings.PROPOSAL_QUEUE_ID + '-'):
        return link
    proposal_ticket_url = '{st_url}/{queue}-'.format(
        st_url=settings.STARTREK_URL,
        queue=settings.PROPOSAL_QUEUE_ID,
    )
    if link.startswith(proposal_ticket_url):
        ticket_no = link.split(settings.PROPOSAL_QUEUE_ID + '-')[1]
        return '{queue}-{no}'.format(queue=settings.PROPOSAL_QUEUE_ID, no=ticket_no)

    raise ValueError('Wrong ticket link')


def clean_name(name):
    return re.sub(r'\s+', ' ', name, flags=re.UNICODE).strip()
