from functools import reduce
import logging
import re

import sform
from staff.lib import waffle
import yenv

from django.conf import settings
from django.core import exceptions
from django import forms
from django.db.models import Q
from django.utils.encoding import smart_str

from staff.lib.utils.ordered_choices import OrderedChoices

from staff.departments.controllers.proposal_action import PERSON_ACTION_SECTIONS
from staff.departments.models import Department
from staff.groups.models import Group, GROUP_TYPE_CHOICES
from staff.map.models import Office
from staff.oebs.models import Job
from staff.payment.enums import WAGE_SYSTEM
from staff.payment.models import Currency
from staff.person.models import Occupation, Organization, Staff

from staff.proposal.forms.base import (
    actions_matcher,
    readonly_if_locked,
    GridFieldWithMeta,
    ProposalBaseForm,
    ProposalEntityActionForm,
)
from staff.proposal.forms.fields import ValueStreamField, GeographyField
from staff.proposal.forms.validators import StarTrekLoginExistenceValidator


logger = logging.getLogger(__name__)


GRADE_DIFF_CHOICES = OrderedChoices(
    ('-2', '-2'),
    ('-1', '-1'),
    ('0', '0'),
    ('+1', '+1'),
    ('+2', '+2'),
)

GRADE_DIFFS = OrderedChoices(
    ('-', '-'),
    ('8', '8'),
    ('9', '9'),
    ('10', '10'),
    ('11', '11'),
    ('12', '12'),
    ('13', '13'),
    ('14', '14'),
    ('15', '15'),
    ('16', '16'),
    ('17', '17'),
    ('18', '18'),
    ('19', '19'),
    ('20', '20'),
)

vacancy_ticket_pattern = re.compile(
    r'^%s/JOB-\d+$' % settings.STARTREK_URL
    if yenv.type == 'production'
    else r'^%s/TJOB-\d+$' % settings.STARTREK_URL
)
vacancy_ticket_key_pattern = re.compile(
    r'^JOB-\d+$'
    if yenv.type == 'production'
    else r'^TJOB-\d+$'
)
vacancies_pattern = re.compile(
    r'^https://{femida_host}/vacancies/.*'.format(femida_host=settings.FEMIDA_HOST)
)


class CurrencyField(sform.Field):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

    @property
    def type_name(self):
        return 'choice'

    @property
    def choices(self):
        currencies_codes = list(Currency.objects.active().values_list('code', flat=True))
        return list(zip(currencies_codes, currencies_codes))

    def get_dj_field(self, required=False):
        return forms.ChoiceField(required=required, choices=self.choices, **self.kwargs)

    def structure_as_dict(self, *args, **kwargs):
        field_dict = super().structure_as_dict(*args, **kwargs)
        field_dict['choices'] = [
            {'value': v, 'label': n} for v, n in self.choices
        ]

        return field_dict


class PersonSalaryForm(ProposalBaseForm):
    old_currency = CurrencyField(
        state=sform.REQUIRED,
        default='RUB',
    )
    new_currency = CurrencyField(
        state=sform.REQUIRED,
        default='RUB',
    )
    old_salary = sform.DecimalField(state=sform.REQUIRED, min_value=0.0, decimal_places=3)
    old_wage_system = sform.ChoiceField(
        state=sform.NORMAL,
        choices=WAGE_SYSTEM.choices(),
        default=WAGE_SYSTEM.FIXED,
    )
    old_rate = sform.DecimalField(state=sform.NORMAL, min_value=0.0, max_value=1.0, decimal_places=3)

    new_salary = sform.DecimalField(state=sform.REQUIRED, min_value=0.0, decimal_places=3)
    new_wage_system = sform.ChoiceField(
        state=sform.NORMAL,
        choices=WAGE_SYSTEM.choices(),
        default=WAGE_SYSTEM.FIXED,
    )
    new_rate = sform.DecimalField(state=sform.NORMAL, min_value=0.0, max_value=1.0, decimal_places=3)

    def clean_old_rate(self, value):
        wage_system = self.cleaned_data.get('old_wage_system', WAGE_SYSTEM.FIXED)
        if wage_system == WAGE_SYSTEM.FIXED and not value:
            raise sform.ValidationError(
                'salary rate required when using fixed wage system',
                code='required',
            )
        return value

    def clean_new_rate(self, value):
        wage_system = self.cleaned_data.get('new_wage_system', WAGE_SYSTEM.FIXED)
        if wage_system == WAGE_SYSTEM.FIXED and not value:
            raise sform.ValidationError(
                'salary rate required when using fixed wage system',
                code='required',
            )
        return value

    def clean(self):
        cleaned_data = self.cleaned_data
        for decimal_field in ('old_salary', 'old_rate', 'new_salary', 'new_rate'):
            value = cleaned_data.get(decimal_field)
            cleaned_data[decimal_field] = value if value is None else str(value)

        return cleaned_data


class SafeSuggestField(sform.SuggestField):
    """ Не 500 для старых заявок """
    type_name = 'staff-suggest'

    def _custom_caption(self, model):
        return reduce(
            lambda total, attr_name: total or getattr(model, attr_name),
            self.label_fields,
            None
        )

    def data_as_dict(self, *args, **kwargs):
        field_dict = super(sform.SuggestField, self).data_as_dict(*args, **kwargs)
        value = kwargs.get(smart_str('value'))
        try:
            model = Job.objects.get(code=value)
        except ValueError:
            # попробуем поискать по name и name_en
            qs = Job.objects.filter(is_deleted_from_oebs=False).filter(Q(name__iexact=value) | Q(name_en__iexact=value))
            model = qs.first() if qs.count() == 1 else None
            if model:
                field_dict['value'] = model.code
        field_dict['caption'] = self._custom_caption(model) if model else value
        return field_dict


class PersonPositionForm(ProposalBaseForm):
    new_position = sform.CharField(state=sform.REQUIRED)
    position_legal = SafeSuggestField(
        queryset=Job.objects.filter(is_deleted_from_oebs=False),
        to_field_name='code',
        label_fields=('name', 'name_en', 'code'),
        state=sform.REQUIRED,
    )

    def clean_position_legal(self, job):
        return job.code


class PersonGradeForm(ProposalBaseForm):
    new_grade = sform.ChoiceField(
        state=sform.REQUIRED,
        choices=GRADE_DIFF_CHOICES.choices(),
        default=GRADE_DIFF_CHOICES['0']
    )
    occupation = sform.ModelChoiceField(
        state=sform.NORMAL,
        queryset=Occupation.objects.order_by('description'),
        to_field_name='name',
        label_extractor='description',
    )
    force_recalculate_schemes = sform.BooleanField(default=False)

    def clean_occupation(self, value):
        return self._clean_hr_analyst_field('occupation', value)

    def clean_force_recalculate_schemes(self, value):
        return self._clean_hr_analyst_field('force_recalculate_schemes', value)

    def structure_as_dict(self, prefix=''):
        result = super().structure_as_dict(prefix)
        author_user = self.base_initial.get('author_user')
        if not author_user or not self._has_hr_analyst_role(author_user.get_profile()):
            result.pop('occupation', None)
            result.pop('force_recalculate_schemes', None)
        return result


class PersonDepartmentForm(ProposalBaseForm):
    with_budget = sform.BooleanField(default=True)
    from_maternity_leave = sform.BooleanField(default=False)
    vacancy_url = sform.CharField(max_length=55)
    department = sform.SuggestField(
        queryset=Department.objects.filter(intranet_status=1),
        to_field_name='url',
        label_fields=('name', 'url'),
    )
    fake_department = sform.CharField(max_length=32)
    service_groups = sform.MultipleSuggestField(
        queryset=Group.objects.filter(
            intranet_status=1,
            type=GROUP_TYPE_CHOICES.SERVICE,
        ),
        to_field_name='url',
        label_fields={'caption': ('name',)},
        state=sform.NORMAL,
    )
    changing_duties = sform.NullBooleanField(state=sform.REQUIRED)

    @readonly_if_locked
    def get_field_state(self, name):
        if name == 'department':
            if not self.data.get('fake_department'):
                return sform.REQUIRED

        return super(PersonDepartmentForm, self).get_field_state(name)

    @staticmethod
    def clean_department(department):
        return department and department.url

    @staticmethod
    def clean_service_groups(service_groups):
        return [sg.url for sg in service_groups]

    def clean_vacancy_url(self, vacancy_url):
        cd = self.cleaned_data

        vacancy_url_is_required = (
            not cd.get('with_budget')
            or cd.get('from_maternity_leave')
        )
        if vacancy_url_is_required and not vacancy_url:
            raise sform.ValidationError(
                'vacancy_url is required',
                code='required',
                params={},
            )

        if vacancy_url and not self.validate_vacancy_url(vacancy_url):
            raise exceptions.ValidationError(
                'Error value of vacancy url provided',
                code='invalid',
            )
        if vacancy_ticket_key_pattern.match(vacancy_url):
            vacancy_url = '{host}/{key}'.format(
                host=settings.STARTREK_URL,
                key=vacancy_url,
            )

        return vacancy_url

    @staticmethod
    def validate_vacancy_url(url):
        vacancy_matches = (
            vacancies_pattern.match(url)
            or vacancy_ticket_pattern.match(url)
            or vacancy_ticket_key_pattern.match(url)
        )
        if vacancy_matches:
            return True

        return False

    def clean(self):
        cleaned_data = self.cleaned_data

        real_dep = cleaned_data.get('department')
        fake_dep = cleaned_data.get('fake_department')
        if not (real_dep or fake_dep):
            self._errors[('department',)] = [
                sform.ValidationError(
                    'department or fake_department is required',
                    code='required',
                    params={},
                )
            ]
        if real_dep and fake_dep:
            self._errors[('department',)] = [
                sform.ValidationError(
                    'department or fake_department have to be filled, not both',
                    code='required',
                    params={},
                )
            ]
        return cleaned_data


class PersonValuestreamForm(ProposalBaseForm):
    value_stream = ValueStreamField(
        queryset=Department.valuestreams.filter(intranet_status=1, hrproduct__intranet_status=1),
        to_field_name='url',
        label_fields=('name', 'url'),
    )

    @staticmethod
    def clean_value_stream(value_stream):
        return value_stream and value_stream.url


class PersonGeographyForm(ProposalBaseForm):
    geography = GeographyField(
        queryset=Department.geography.filter(intranet_status=1, geography_instance__oebs_code__isnull=False),
        to_field_name='url',
        label_fields=('name', 'url'),
    )

    @staticmethod
    def clean_geography(geography):
        return geography and geography.url


class PersonOfficeForm(ProposalBaseForm):
    office = sform.ModelChoiceField(
        queryset=Office.objects.filter(intranet_status=1).order_by('name'),
        label_extractor='name',
        state=sform.REQUIRED,
    )

    @staticmethod
    def clean_office(office):
        return office and office.id


class PersonOrganizationForm(ProposalBaseForm):
    organization = sform.ModelChoiceField(
        queryset=Organization.objects.filter(intranet_status=1).order_by('name'),
        label_extractor='name',
        state=sform.REQUIRED,
    )

    @staticmethod
    def clean_organization(organization):
        return organization and organization.id


class PersonEditForm(ProposalEntityActionForm):
    sections = sform.MultipleChoiceField(choices=PERSON_ACTION_SECTIONS.choices(), state=sform.REQUIRED)
    salary = sform.FieldsetField(PersonSalaryForm, trim_on_empty=True)
    position = sform.FieldsetField(PersonPositionForm, trim_on_empty=True)
    grade = sform.FieldsetField(PersonGradeForm, trim_on_empty=True)
    department = sform.FieldsetField(PersonDepartmentForm, trim_on_empty=True)
    office = sform.FieldsetField(PersonOfficeForm, trim_on_empty=True)
    organization = sform.FieldsetField(PersonOrganizationForm, trim_on_empty=True)
    value_stream = sform.FieldsetField(PersonValuestreamForm, trim_on_empty=True)
    geography = sform.FieldsetField(PersonGeographyForm, trim_on_empty=True)
    comment = sform.CharField(max_length=2048)
    login = sform.CharField(state=sform.REQUIRED, max_length=22)

    @staticmethod
    def _is_section_field(name):
        return name in [text_value for db_value, text_value in PERSON_ACTION_SECTIONS]

    def _section_enabled(self, name):
        return self.data and name in self.data.get('sections', [])

    def prepare_fields_state(self):
        if not waffle.switch_is_active('enable_proposal_geography'):
            self.fields.pop('geography')
            self.fields['sections'].choices.remove(('geography', 'geography'))

        return super().prepare_fields_state()

    @readonly_if_locked
    def get_field_state(self, name):
        if self._is_section_field(name) and self._section_enabled(name):
            return sform.REQUIRED

        return super(ProposalBaseForm, self).get_field_state(name)

    def clean_comment(self, comment_value):
        comment_value = comment_value.strip()
        if {'salary', 'position', 'grade'} & set(self.cleaned_data) and not comment_value:
            raise sform.ValidationError(
                'Comment required',
                code='required',
            )
        return comment_value

    def clean_login(self, login):
        try:
            person = Staff.objects.get(login=login)
        except Staff.DoesNotExist:
            raise sform.ValidationError(
                'Login `{login}` does not exist'.format(login=login),
                code='invalid',
            )

        if person.is_dismissed:
            raise sform.ValidationError(f'Person `{login}` is dismissed', code='person_is_dismissed')

        cleaned_data = self.cleaned_data
        office = cleaned_data.get('office')
        if office and office.get('office') == person.office_id:
            self._errors[('office',)] = [
                sform.ValidationError(
                    'Office has not been changed',
                    code='invalid',
                )
            ]
        return login

    def clean(self):
        cleaned_data = self.cleaned_data
        if not cleaned_data:
            raise sform.ValidationError(
                'At least one block is required',
                code='required',
            )

        return cleaned_data


class PersonGridFieldWithMeta(GridFieldWithMeta):

    def populate_with_meta(self, fields_list, base_initial):
        from staff.proposal.controllers import PersonDataCtl
        logins = [
            action['value']['login']['value']
            for action in fields_list['value']
        ]

        person_data_ctl = PersonDataCtl([login for login in logins if login])

        for field_dict in fields_list['value']:
            if field_dict['value']['login']['value']:
                field_dict['meta'] = person_data_ctl.as_meta(field_dict['value']['login']['value'])
            else:
                field_dict['meta'] = {}


class PersonsChangesProposalForm(ProposalBaseForm):
    actions = PersonGridFieldWithMeta(
        sform.FieldsetField(PersonEditForm),
        values_matcher=actions_matcher,
    )

    def clean(self):
        cleaned_data = self.cleaned_data
        if not self.is_valid():
            return cleaned_data

        logins = [action['login'] for action in cleaned_data['actions']]
        StarTrekLoginExistenceValidator().validate_logins(logins)

        return cleaned_data
