from datetime import date
from typing import Dict, List, Iterator

from staff.departments.controllers.actions.person_action import PersonAction
from staff.departments.models import DEADLINE_TYPE
from staff.map.models import Office
from staff.person.models import Staff

from staff.proposal.hr_deadlines.nearest_deadline import NearestDeadline, Deadline


class SplitData:

    def __init__(
        self,
        proposal_date: date,
        desired_date: date,
        deadlines: Dict[str, Deadline],
        office_to_city: Dict[int, int],
        login_to_cur_city: Dict[str, int],
        login_to_paysys: Dict[str, str]
    ) -> None:
        self.proposal_date = proposal_date
        self.desired_date = desired_date
        self.office_to_city = office_to_city
        self.login_to_cur_city = login_to_cur_city
        self.login_to_paysys = login_to_paysys
        self.future_date: date = deadlines[DEADLINE_TYPE.FUTURE_CHANGE].month
        self.structure_date: date = deadlines[DEADLINE_TYPE.STRUCTURE_CHANGE].month
        self.salary_date: date = deadlines[DEADLINE_TYPE.SALARY_CHANGE].month
        self.table_date: date = deadlines[DEADLINE_TYPE.TABLE_TRANSFER].month
        self.deal_date: date = deadlines[DEADLINE_TYPE.DEAL_DATE].month

    @classmethod
    def from_person_actions(
        cls,
        person_actions: List[PersonAction],
        proposal_date: date,
        desired_date: date,
        nearest_deadline: NearestDeadline,
    ) -> 'SplitData':
        deadlines = nearest_deadline.find(proposal_date)
        return cls(
            proposal_date,
            desired_date,
            deadlines=deadlines,
            office_to_city=cls._get_office_to_city(person_actions),
            **cls._get_person_info(person_actions)
        )

    @staticmethod
    def _get_office_to_city(person_actions: Iterator[PersonAction]) -> Dict[int, int]:
        office_ids = [it.office for it in person_actions if it.office]
        if office_ids:
            return dict(
                Office.objects
                .filter(id__in=office_ids)
                .values_list('id', 'city_id')
            )
        return {}

    @staticmethod
    def _get_person_info(person_actions: Iterator[PersonAction]) -> Dict[str, Dict]:
        staff_q = (
            Staff.objects
            .filter(login__in=[it.login for it in person_actions])
            .values_list('login', 'office__city_id', 'extra__oebs_payment_system_id')
        )
        login_to_paysys = {}
        login_to_cur_city = {}
        for login, city_id, paysys in staff_q:
            login_to_paysys[login] = paysys
            login_to_cur_city[login] = city_id
        return {'login_to_cur_city': login_to_cur_city, 'login_to_paysys': login_to_paysys}
