from collections import defaultdict
from datetime import date
from itertools import chain
from typing import Dict, Iterator, List

from staff.departments.controllers.actions.person_action import PersonAction

from staff.proposal.hr_deadlines.deadline_matcher import DeadlineMatcher
from staff.proposal.hr_deadlines.nearest_deadline import NearestDeadline
from staff.proposal.hr_deadlines.split_data import SplitData


def split_person_actions(person_actions: Iterator[Dict], proposal_date: date, desired_date: date) -> List[Dict]:
    person_action_objs = [*map(PersonAction.from_dict, person_actions)]
    split_data = SplitData.from_person_actions(person_action_objs, proposal_date, desired_date, NearestDeadline())
    res = chain.from_iterable(split(act, split_data) for act in person_action_objs)
    return [it.as_dict() for it in res]


def split(
    action: PersonAction,
    split_data: SplitData,
    matcher: DeadlineMatcher = None,
) -> Iterator[PersonAction]:
    matcher = matcher or DeadlineMatcher()
    date_to_change = defaultdict(dict)

    if action.organization:
        date_to_change[split_data.future_date]['organization'] = action.organization

    if action.office:
        date_to_change[split_data.future_date]['office'] = action.office

    if action.has_salary_change():
        date_to_change[matcher.get_salary_date(action, split_data)]['salary'] = action.salary

    if action.has_dep_change():
        new_date = matcher.get_pieceworker_date(action.login, split_data, split_data.structure_date)
        date_to_change[new_date]['department'] = action.department

    if action.has_grade_change():
        date_to_change[split_data.salary_date]['grade'] = action.grade

    if action.has_position_change():
        new_date = matcher.get_pieceworker_date(action.login, split_data, split_data.structure_date)
        date_to_change[new_date]['position'] = action.position

    if action.has_valuestream_change():
        new_date = matcher.get_pieceworker_date(action.login, split_data, split_data.table_date)
        date_to_change[new_date]['value_stream'] = action.value_stream

    if action.has_geography_change():
        new_date = matcher.get_pieceworker_date(action.login, split_data, split_data.table_date)
        date_to_change[new_date]['geography'] = action.geography

    corrected_data = matcher.correct_application_dates(action, date_to_change, split_data)

    return (
        action.copy_with_meta(oebs_application_date=date_, **kw)
        for date_, kw in corrected_data.items()
    )
