from datetime import date
from random import randrange
from typing import Callable, Dict, List, Optional, Tuple

from assertpy import assert_that

from staff.departments.controllers.proposal import (
    ProposalCtl,
    ProposalApprovement,
    ApprovementTicketType,
)
from staff.departments.controllers.actions.department_action import DepartmentAction
from staff.departments.controllers.actions.headcount_action import HeadcountAction
from staff.departments.controllers.actions.person_action import PersonAction
from staff.departments.controllers.actions.vacancy_action import VacancyAction
from staff.departments.models import Department, DepartmentKind
from staff.person.models import Staff


class PersonActionBuilder:
    @classmethod
    def with_move_to_new_dep(cls, login: str, fake_id: str):
        return cls(login, department_url=fake_id, new_department=True)

    @classmethod
    def with_move_to_existing_dep(cls, login: str, url: str):
        return cls(login, department_url=url, new_department=False)

    @classmethod
    def without_move(cls, login: str):
        return cls(login)

    def __init__(self, login: str, department_url: str = None, new_department: bool = None):
        self.new_staff_position = None
        self.position_legal: Optional[int] = None  # Job.id
        self.person_ticket = None
        self.organization_id: Optional[int] = None
        self.occupation_name: Optional[str] = None
        self.new_grade: Optional[str] = None
        self.force_recalculate_schemes: bool = False
        self.new_salary_value: Optional[str] = None
        self.old_salary_value: Optional[str] = None
        self.login = login
        self.new_value_stream: str = None
        self._department_url = department_url
        self._new_department = new_department
        self._sections = set()

    def staff_position(self, new_staff_position) -> 'PersonActionBuilder':
        self.new_staff_position = new_staff_position
        self._sections.add('position')
        return self

    def organization(self, organization_id: int) -> 'PersonActionBuilder':
        self.organization_id = organization_id
        self._sections.add('organization')
        return self

    def occupation(self, occupation_name: str) -> 'PersonActionBuilder':
        self.occupation_name = occupation_name
        self._sections.add('grade')
        return self

    def grade(self, new_grade: str) -> 'PersonActionBuilder':
        self.new_grade = new_grade
        self._sections.add('grade')
        return self

    def force_recalculate_schemes(self, force_recalculate_schemes: bool) -> 'PersonActionBuilder':
        self.force_recalculate_schemes = force_recalculate_schemes
        self._sections.add('grade')
        return self

    def legal_position(self, new_legal_position: int) -> 'PersonActionBuilder':
        self.position_legal = new_legal_position
        self._sections.add('position')
        return self

    def new_salary(self, new_salary: str, old_salary: str) -> 'PersonActionBuilder':
        self.new_salary_value = new_salary
        self.old_salary_value = old_salary
        self._sections.add('salary')
        return self

    def value_stream(self, value_stream_url: str) -> 'PersonActionBuilder':
        self.new_value_stream = value_stream_url
        self._sections.add('value_stream')
        return self

    def with_ticket(self, ticket_key: str):
        self.person_ticket = ticket_key
        return self

    def build(self) -> Tuple[PersonAction, str]:
        assert self.login, 'You have to specify user login to change'

        result = PersonAction(
            action_id=f'act_{randrange(10000, 100000)}',
            login=self.login,
            sections=list(self._sections),
            department=None,
        )

        if self.new_staff_position:
            result.position.new_position = self.new_staff_position

        if self.position_legal:
            result.position.position_legal = self.position_legal

        if self._department_url:
            result.department = PersonAction.DepartmentChange()
            result.sections.append('department')
            if self._new_department:
                result.department.fake_department = self._department_url
            else:
                result.department.department = self._department_url

        if self.organization_id:
            result.organization = self.organization_id

        if self.new_salary_value:
            result.salary.new_salary = self.new_salary_value
            result.salary.old_salary = self.old_salary_value

        if self.occupation_name:
            result.grade.occupation = self.occupation_name

        if self.new_grade:
            result.grade.new_grade = self.new_grade

        if self.force_recalculate_schemes is not None:
            result.grade.force_recalculate_schemes = self.force_recalculate_schemes

        if self.new_value_stream:
            result.value_stream.value_stream = self.new_value_stream

        return result, self.person_ticket


PersonConfigCallable = Callable[[PersonActionBuilder], PersonActionBuilder]


class VacancyActionBuilder:
    @classmethod
    def with_move_to_new_dep(cls, vacancy_id: int, fake_id: str):
        return cls(vacancy_id, department_url=fake_id, new_department=True)

    @classmethod
    def with_move_to_existing_dep(cls, vacancy_id: int, url: str):
        return cls(vacancy_id, department_url=url, new_department=False)

    @classmethod
    def without_move(cls, vacancy_id: int):
        return cls(vacancy_id)

    def __init__(self, vacancy_id: int, department_url: str = None, new_department: bool = None):
        self.vacancy_ticket = None
        self.vacancy_id = vacancy_id
        self._department_url = department_url
        self._new_department = new_department
        self._new_value_stream: str or None = None

    def with_ticket(self, ticket_key):
        self.vacancy_ticket = ticket_key
        return self

    def value_stream(self, value_stream_url: str) -> 'VacancyActionBuilder':
        self._new_value_stream = value_stream_url
        return self

    def build(self) -> Tuple[VacancyAction, str]:
        assert self.vacancy_id, 'You have to specify vacancy_id to change'

        result = VacancyAction(
            action_id=f'act_{randrange(10000, 100000)}',
            vacancy_id=self.vacancy_id,
        )
        if self._department_url:
            if self._new_department:
                result.fake_department = self._department_url
            else:
                result.department = self._department_url

        if self._new_value_stream:
            result.value_stream = self._new_value_stream

        return result, self.vacancy_ticket


VacancyConfigCallable = Callable[[VacancyActionBuilder], VacancyActionBuilder]


class HeadcountActionBuilder:
    @classmethod
    def with_move_to_new_dep(cls, headcount_code: int, fake_id: str) -> 'HeadcountActionBuilder':
        return cls(headcount_code, department_url=fake_id, new_department=True)

    @classmethod
    def with_move_to_existing_dep(cls, headcount_code: int, url: str) -> 'HeadcountActionBuilder':
        return cls(headcount_code, department_url=url, new_department=False)

    @classmethod
    def without_move(cls, headcount_code: int) -> 'HeadcountActionBuilder':
        return cls(headcount_code)

    def __init__(self, headcount_code: int, department_url: str = None, new_department: bool = None):
        self.headcount_ticket = None
        self.headcount_code = headcount_code
        self._department_url = department_url
        self._new_department = new_department
        self._new_value_stream: str or None = None

    def with_ticket(self, ticket_key) -> 'HeadcountActionBuilder':
        self.headcount_ticket = ticket_key
        return self

    def value_stream(self, value_stream_url) -> 'HeadcountActionBuilder':
        self._new_value_stream = value_stream_url
        return self

    def build(self) -> Tuple[HeadcountAction, str]:
        assert self.headcount_code is not None, 'You have to specify headcount code to change'

        result = HeadcountAction(action_id=f'act_{randrange(10000, 100000)}', headcount_code=self.headcount_code)

        if self._department_url:
            if self._new_department:
                result.fake_department = self._department_url
            else:
                result.department = self._department_url

        if self._new_value_stream:
            result.value_stream = self._new_value_stream

        return result, self.headcount_ticket


HeadcountConfigCallable = Callable[[HeadcountActionBuilder], HeadcountActionBuilder]


class DepartmentActionBuilder:
    @classmethod
    def for_new_department(cls, name: str) -> 'DepartmentActionBuilder':
        return cls(fake_id=f'fake_{randrange(10000, 100000)}', name=name)

    @classmethod
    def for_existing_department(cls, department_url: str) -> 'DepartmentActionBuilder':
        return cls(url=department_url)

    def __init__(self, fake_id: str = None, url: str = None, name: str = None):
        assert bool(fake_id) ^ bool(url)
        self.url: str = url
        self.fake_id: str = fake_id

        self.parent_url: str = None
        self.fake_parent_id: str = None

        self.name: str = name
        self.administration: DepartmentAction.AdministrationChange or None = None

        self._person_actions: List[Tuple[PersonAction, str]] = []
        self._vacancy_actions: List[Tuple[VacancyAction, str]] = []
        self._headcount_actions: List[Tuple[HeadcountAction, str]] = []

    def set_parent(self, department_url: str) -> 'DepartmentActionBuilder':
        self.parent_url = department_url
        return self

    def set_name(self, name: str) -> 'DepartmentActionBuilder':
        self.name = name
        return self

    def use_for_person(self, login: str, person_conf: PersonConfigCallable = None) -> 'DepartmentActionBuilder':
        person_builder = (
            PersonActionBuilder.with_move_to_new_dep(login, self.fake_id)
            if self._creates_new_department()
            else PersonActionBuilder.with_move_to_existing_dep(login, self.url)
        )

        if person_conf:
            person_builder = person_conf(person_builder)

        self._person_actions.append(person_builder.build())
        return self

    def change_roles(self, chief_id: int or None, deputies: List[int] or None) -> 'DepartmentActionBuilder':
        self.administration = DepartmentAction.AdministrationChange(chief=chief_id, deputies=deputies or [])
        return self

    def use_for_vacancy(self, vacancy_id: int, vacancy_conf: VacancyConfigCallable = None) -> 'DepartmentActionBuilder':
        vacancy_builder = (
            VacancyActionBuilder.with_move_to_new_dep(vacancy_id, self.fake_id)
            if self._creates_new_department()
            else VacancyActionBuilder.with_move_to_existing_dep(vacancy_id, self.url)
        )

        if vacancy_conf:
            vacancy_builder = vacancy_conf(vacancy_builder)

        self._vacancy_actions.append(vacancy_builder.build())
        return self

    def use_for_headcount(
        self,
        headcount_code: int,
        headcount_conf: HeadcountConfigCallable = None,
    ) -> 'DepartmentActionBuilder':
        headcount_builder = (
            HeadcountActionBuilder.with_move_to_new_dep(headcount_code, self.fake_id)
            if self._creates_new_department()
            else HeadcountActionBuilder.with_move_to_existing_dep(headcount_code, self.url)
        )

        if headcount_conf:
            headcount_builder = headcount_conf(headcount_builder)

        self._headcount_actions.append(headcount_builder.build())
        return self

    def _creates_new_department(self) -> bool:
        return bool(self.fake_id)

    def build(self) -> Tuple[
        List[DepartmentAction],
        List[Tuple[PersonAction, str]],
        List[Tuple[VacancyAction, str]],
        List[Tuple[HeadcountAction, str]],
    ]:
        result = DepartmentAction()

        if self.name:
            result.name = DepartmentAction.NameChange(name=self.name, name_en=self.name)

        if self._creates_new_department():
            assert self.name, 'You have to set name for new department'
            assert self.parent_url, 'You have to set parent for new department'
            result.fake_id = self.fake_id
            result.hierarchy = DepartmentAction.HierarchyChange(
                parent=Department.objects.get(url=self.parent_url).id
            )
            result.technical = DepartmentAction.TechnicalChange(
                code=self.fake_id,
                kind=DepartmentKind.objects.get_or_create(
                    slug='department_kind',
                    created_at=date.today(),
                    modified_at=date.today(),
                )[0].id,
                category='technical',
                position=0,
            )

        else:
            result.id = Department.objects.get(url=self.url).id
            if self.parent_url:
                result.hierarchy = DepartmentAction.HierarchyChange(
                    parent=Department.objects.get(url=self.parent_url).id
                )

        if self.administration is not None:
            result.administration = self.administration

        return [result], self._person_actions, self._vacancy_actions, self._headcount_actions


DepartmentConfigCallable = Callable[[DepartmentActionBuilder], DepartmentActionBuilder]


class ProposalBuilder:
    def __init__(self):
        self._departments_actions: List[DepartmentAction] = []
        self._persons_actions: List[PersonAction] = []
        self._persons_tickets: Dict[str, str] = {}
        self._vacancies_actions: List[VacancyAction] = []
        self._vacancies_tickets: Dict[int, str] = {}
        self._headcounts_actions: List[VacancyAction] = []
        self._headcounts_tickets: Dict[int, str] = {}
        self._linked_ticket: str or None = None

    def with_person(self, login: str, person_config: PersonConfigCallable) -> 'ProposalBuilder':
        builder = PersonActionBuilder.without_move(login)
        action, person_ticket = person_config(builder).build()
        self._persons_actions.append(action)
        if person_ticket:
            self._persons_tickets[action.login] = person_ticket
        return self

    def with_vacancy(self, vacancy_id: int, vacancy_config: VacancyConfigCallable) -> 'ProposalBuilder':
        builder = VacancyActionBuilder.without_move(vacancy_id)
        action, vacancy_ticket = vacancy_config(builder).build()
        self._vacancies_actions.append(action)
        if vacancy_ticket:
            self._vacancies_tickets[action.vacancy_id] = vacancy_ticket
        return self

    def with_headcount(self, headcount_code: int, headcount_config: HeadcountConfigCallable) -> 'ProposalBuilder':
        builder = HeadcountActionBuilder.without_move(headcount_code)
        action, headcount_ticket = headcount_config(builder).build()
        self._headcounts_actions.append(action)
        if headcount_ticket:
            self._headcounts_tickets[action.headcount_code] = headcount_code
        return self

    def for_new_department(self, name: str, department_config: DepartmentConfigCallable) -> 'ProposalBuilder':
        department_action_builder = DepartmentActionBuilder.for_new_department(name=name)
        self._for_department(department_action_builder, department_config)
        return self

    def for_existing_department(
        self,
        department_url: str,
        department_config: DepartmentConfigCallable,
    ) -> 'ProposalBuilder':
        department_action_builder = DepartmentActionBuilder.for_existing_department(department_url=department_url)
        self._for_department(department_action_builder, department_config)
        return self

    def with_linked_ticket(self, ticket_key: str) -> 'ProposalBuilder':
        self._linked_ticket = ticket_key
        return self

    def _for_department(self, department_action_builder: DepartmentActionBuilder,
                        department_config: DepartmentConfigCallable) -> None:
        (
            departments_actions,
            persons_actions,
            vacancies_actions,
            headcount_actions,
        ) = department_config(department_action_builder).build()
        self._departments_actions += departments_actions

        for person_action, person_ticket in persons_actions:
            self._persons_actions.append(person_action)
            if person_ticket:
                self._persons_tickets[person_action.login] = person_ticket

        for vacancy_action, vacancy_ticket in vacancies_actions:
            self._vacancies_actions.append(vacancy_action)
            if vacancy_ticket:
                self._vacancies_tickets[vacancy_action.vacancy_id] = vacancy_ticket

        for headcount_action, headcount_ticket in headcount_actions:
            self._headcounts_actions.append(headcount_action)
            if headcount_ticket:
                self._vacancies_tickets[headcount_action.headcount_code] = headcount_ticket

    def delete_department(self, department_url) -> 'ProposalBuilder':
        department_action = DepartmentAction(
            delete=True,
            id=Department.objects.get(url=department_url).id,
        )

        self._departments_actions.append(department_action)
        return self

    def build(self, author_login: str) -> str:
        (
            assert_that(self._persons_actions).extracting('login')
            .does_not_contain_duplicates()
            .described_as('You can\'t make two actions for the same person')
        )
        (
            assert_that(self._vacancies_actions).extracting('vacancy_id')
            .does_not_contain_duplicates()
            .described_as('You can\'t make two actions for the same vacancy')
        )
        (
            assert_that(self._headcounts_actions).extracting('headcount_code')
            .does_not_contain_duplicates()
            .described_as('You can\'t make two actions for the same headcount')
        )
        (
            assert_that(self._departments_actions).extracting('id', filter='id')
            .does_not_contain_duplicates()
            .described_as('You can\'t make two actions for the same department')
        )

        approvements = []
        type_to_tickets = {
            ApprovementTicketType.VACANCY: self._vacancies_tickets.values(),
            ApprovementTicketType.HEADCOUNT: self._headcounts_tickets.values(),
            ApprovementTicketType.PERSONAL: self._persons_tickets.values(),
        }
        for ticket_type, tickets in type_to_tickets.items():
            approvements += [
                ProposalApprovement(key, ticket_type).as_dict()
                for key in tickets
            ]

        _proposal_data = {
            'tickets': {
                'persons': self._persons_tickets,
                'vacancies': self._vacancies_tickets,
                'headcounts': self._headcounts_tickets,
            },
            'persons': {
                'actions': [action.as_dict() for action in self._persons_actions]
            },
            'vacancies': {
                'actions': [action.as_dict() for action in self._vacancies_actions]
            },
            'headcounts': {
                'actions': [action.as_dict() for action in self._headcounts_actions],
            },
            'actions': [action.as_dict() for action in self._departments_actions],
            'apply_at': date.today(),
            'approvements': approvements,
        }

        if self._linked_ticket:
            _proposal_data['tickets']['department_linked_ticket'] = self._linked_ticket

        author = Staff.objects.get(login=author_login)
        proposal_ctl = ProposalCtl(author=author).create_from_cleaned_data(_proposal_data)
        proposal_ctl.save()
        return proposal_ctl.proposal_id
