from django.http import JsonResponse
from django.views.decorators.csrf import ensure_csrf_cookie
from django.views.decorators.http import require_http_methods

from staff.departments.models import Department
from staff.lib.decorators import responding_json
from staff.lib.forms import errors as form_errors

from staff.proposal.controllers import DepartmentDataCtl, PersonDataCtl
from staff.proposal.controllers.headcount import HeadcountDataCtl
from staff.proposal.controllers.vacancy import VacancyDataCtl
from staff.proposal.forms.department import DepartmentEditForm, find_conflict_proposals
from staff.proposal.forms.headcount import HeadcountEditForm
from staff.proposal.forms.person import PersonEditForm
from staff.proposal.forms.vacancy import VacancyEditForm


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
def department_data(request):
    """Данные подразделений для предзаполнения полей"""
    department_urls = (
        Department.objects
        .filter(url__in=request.GET.getlist('url'))
        .values_list('url', flat=True)
    )
    if not department_urls:
        return form_errors.sform_general_error(
            err_code='department_url_not_specified',
            params={''},
        ), 400

    person = request.user.get_profile()
    data_controller = DepartmentDataCtl(department_urls, person)
    dep_data = {
        url: {
            'value': DepartmentEditForm(
                initial=data_controller.as_form_data(url)
            ).data_as_dict(prefix='departments[actions][0]'),
            'meta': data_controller.as_meta(url),
        }
        for url in department_urls
    }
    for url, data in dep_data.items():
        conflicting_proposals = find_conflict_proposals(url)
        if conflicting_proposals:
            data['errors'] = form_errors.sform_general_error(
                err_code='department_conflict',
                params={'proposal_uid': conflicting_proposals[0]},
            )['errors']['']

    return dep_data


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
def person_data(request):
    """Данные сотрудников для предзаполнения полей"""
    logins = request.GET.getlist('login')
    if not logins:
        return form_errors.sform_general_error(
            err_code='logins_not_specified',
            params={''},
        ), 400

    data_controller = PersonDataCtl(logins)

    return {
        login: {
            'value': PersonEditForm(
                initial=data_controller.as_form_data(login)
            ).data_as_dict(prefix='persons[actions][0]'),
            'meta': data_controller.as_meta(login),
        }
        for login in logins
    }


@ensure_csrf_cookie
@require_http_methods(['GET'])
def headcount_data(request):
    """Данные hc для предзаполнения полей"""
    observer = request.user.get_profile()
    headcount_codes = request.GET.getlist('headcount_code')

    if not headcount_codes:
        return JsonResponse(
            data=form_errors.sform_general_error(err_code='headcount_code_not_specified', params=''),
            status=400,
        )

    try:
        headcount_codes = [int(headcount_code) for headcount_code in headcount_codes]
    except ValueError:
        return JsonResponse(
            data=form_errors.sform_general_error(err_code='headcount_code_error', params=''),
            status=400,
        )

    data_controller = HeadcountDataCtl(headcount_codes, observer)
    result = {
        code: {
            'value': HeadcountEditForm(
                initial=data_controller.as_form_data(code)
            ).data_as_dict(prefix='headcounts[actions][0]'),
            'meta': data_controller.as_meta(code),
        }
        for code in headcount_codes
    }

    return JsonResponse(data=result)


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
def vacancy_data(request):
    """Данные сотрудников для предзаполнения полей"""
    vacancy_ids = request.GET.getlist('vacancy_id')

    if not vacancy_ids:
        return form_errors.sform_general_error(
            err_code='vacancy_ids_not_specified',
            params={''},
        ), 400

    try:
        vacancy_ids = [int(v_id) for v_id in vacancy_ids]
    except ValueError:
        return form_errors.sform_general_error(
            err_code='vacancy_ids_error',
            params='',
        ), 400

    data_controller = VacancyDataCtl(vacancy_ids, request.user.get_profile())

    return {
        vacancy_id: {
            'value': VacancyEditForm(
                initial=data_controller.as_form_data(vacancy_id)
            ).data_as_dict(prefix='vacancies[actions][0]'),
            'meta': data_controller.as_meta(vacancy_id),
        }
        for vacancy_id in vacancy_ids
    }


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
def persons_meta(request):
    """Данные сотрудников для предзаполнения полей"""
    logins = [login for login in request.GET.getlist('login') if login]

    if not logins:
        return form_errors.sform_general_error(
            err_code='logins_not_specified',
            params='',
        ), 400

    data_controller = PersonDataCtl(logins)

    return {
        login: data_controller.as_meta(login)
        for login in logins
    }


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
def departments_meta(request):
    """Данные сотрудников для предзаполнения полей"""
    urls = request.GET.getlist('url')
    if not urls:
        return form_errors.sform_general_error(
            err_code='urls_not_specified',
            params='',
        ), 400

    person = request.user.get_profile()
    data_controller = DepartmentDataCtl(urls, person)
    return {
        url: data_controller.as_meta(url)
        for url in urls
    }
