from django.views.decorators.http import require_http_methods

from staff.lib.decorators import responding_json, use_request_lang

from staff.departments.models import Department
from staff.departments.tree.persons_entity_info import ProposalPersonsEntityInfo
from staff.departments.tree.vacancies_entity_info import VacanciesEntityInfo
from staff.departments.tree.vacancies_filter_context import VacanciesFilterContext
from staff.departments.tree_lib import ProposalTreeExpander
from staff.departments.tree_lib.tree_expander import ProposalVacanciesTreeExpander
from staff.departments.vacancy_permissions import VacancyPermissions
from staff.headcounts.permissions import Permissions
from staff.headcounts.positions_entity_info import PositionsEntityInfo
from staff.headcounts.positions_filter_context import PositionsFilterContext
from staff.oebs.constants import PERSON_POSITION_STATUS
from staff.person_filter.controller import FilterNotFound
from staff.person_filter.filter_context import FilterContext


@responding_json
@require_http_methods(['GET'])
@use_request_lang
def load_persons(request, department_url):
    observer = request.user
    filter_id = request.GET.get('filter')
    try:
        filter_context = FilterContext(filter_id=filter_id, observer_tz=observer.get_profile().tz, user=observer)
    except FilterNotFound:
        return {'error': 'filter_not_found'}, 404

    try:
        return ProposalTreeExpander(
            ProposalPersonsEntityInfo(filter_context),
            department_url,
        ).expand()
    except Department.DoesNotExist:
        return {'error': 'department_not_found'}, 404


@responding_json
@require_http_methods(['GET'])
@use_request_lang
def load_vacancies(request, department_url):
    observer = request.user.get_profile()
    permissions = Permissions(observer)
    vacancy_permissions = VacancyPermissions(observer)

    try:
        department = Department.objects.get(url=department_url)
    except Department.DoesNotExist:
        return {'error': 'department_not_found'}, 404

    positions_filter_context = PositionsFilterContext(
        observer_permissions=permissions,
        status__in=(PERSON_POSITION_STATUS.VACANCY_OPEN, PERSON_POSITION_STATUS.OFFER),
        department=department,
    )
    vacancy_filter = {
        'headcount_position_code__in': positions_filter_context.positions_objects_qs().values('code'),
        'budget_position__isnull': False,
    }

    filter_context = VacanciesFilterContext(vacancy_permissions, **vacancy_filter)
    tree_expander = ProposalVacanciesTreeExpander(VacanciesEntityInfo(filter_context), department_url)
    return tree_expander.expand()


@responding_json
@require_http_methods(['GET'])
@use_request_lang
def load_headcounts(request, department_url):
    observer = request.user.get_profile()

    department = Department.objects.filter(url=department_url).first()
    if not department:
        return {'error': 'department_not_found'}, 404

    permissions = Permissions(observer)
    filter_context = PositionsFilterContext(
        observer_permissions=permissions,
        department_url=department_url,
        position_statuses=(PERSON_POSITION_STATUS.VACANCY_PLAN, PERSON_POSITION_STATUS.RESERVE),
    )
    tree_expander = ProposalTreeExpander(PositionsEntityInfo(filter_context), department_url)
    return tree_expander.expand()
