import attr
from IPy import IPint


@attr.s(auto_attribs=True)
class Rule:
    left_begin: int
    left_end: int
    right_begin: int
    right_end: int

    def is_inside(self, another_rule: 'Rule') -> bool:
        return all([
            another_rule.left_begin <= self.left_begin,
            another_rule.left_end >= self.left_end,
            another_rule.right_begin <= self.right_begin,
            another_rule.right_end >= self.right_end,
        ])


class YaIP(IPint):
    def __init__(self, data, ipversion=0, make_net=0):
        self.project_id = None

        if '@' in data:
            self.project_id, data = data.split('@')
            self.right = Segment(
                value=int(self.project_id, 16) << 32,
                fixed_len=32,
            )

        super(YaIP, self).__init__(data, ipversion, make_net)

        if not self.project_id:
            self.right = Segment(
                value=self.net() & ((1 << 64) - 1),
                fixed_len=self._prefixlen + (32 if self.version() == 4 else -64),
            )

        self.left = Segment(
            value=self.net() >> 64,
            fixed_len=self._prefixlen if self.version() == 6 else 64,
        )

    def as_rule(self) -> Rule:
        left_rule = self.left.as_rule()
        right_rule = self.right.as_rule()
        return Rule(left_rule['begin'], left_rule['end'], right_rule['begin'], right_rule['end'])

    def __str__(self):
        ip_as_str = super(YaIP, self).__str__()
        if self.project_id:
            ip_as_str = '@'.join([str(self.project_id), ip_as_str])

        return ip_as_str

    def __repr__(self):
        return "YaIP('%s')" % str(self)


@attr.s
class Segment:
    value = attr.ib(type=int)
    fixed_len = attr.ib(type=int, converter=lambda x: max(min(x, 64), 0))  # type: ignore

    def as_rule(self):
        unfixed_len = 64 - self.fixed_len
        begin = (self.value >> unfixed_len) << unfixed_len
        end = begin + (1 << unfixed_len) - 1
        return {
            'begin': begin,
            'end': end
        }
