from typing import List, Type
import json
import logging
import socket

from django.conf import settings

from staff.celery_app import app

from staff.lib import requests
from staff.lib.db import atomic
from staff.lib.tasks import LockedTask

from staff.racktables.models import AllowedNet, DeniedNet, NetRule
from staff.racktables.objects import YaIP


logger = logging.getLogger(__name__)


def get_ip(name):
    # type: (str) -> List[YaIP]
    try:
        return list(set([YaIP(address[4][0]) for address in socket.getaddrinfo(name, 80)]))
    except socket.gaierror as e:
        if e.args[0] == socket.EAI_NONAME:
            return []
        raise


def get_nets_by_macro(macro):
    # type: (str) -> List[YaIP]

    url = '{HBF_URL}/{_MACRO_NAME_}?[format={FORMAT}]&[trypo_format={TRYPO_FORMAT}]'
    response = requests.get(
        url.format(
            HBF_URL=settings.HBF_URL,
            FORMAT='json',
            TRYPO_FORMAT='trypo',
            _MACRO_NAME_=macro,
        ),
        timeout=(0.3, 1, 5),
    )

    if response.status_code != 200:
        logger.error('Racktables error ({}) when getting nets for {} macro: {}'.format(
            response.status_code,
            macro,
            response.text,
        ))
        return []

    result = []
    for net in json.loads(response.content):
        try:
            result.append(YaIP(net))
        except ValueError:
            result.extend(get_ip(net))
    return result


def collect_rules(model: Type) -> List[YaIP]:
    nets = []
    for net in model.objects.values_list('macro_or_net', flat=True):
        try:
            nets.append(YaIP(net))
        except ValueError:
            nets.extend(get_nets_by_macro(net))
    return nets


@app.task
class SyncAllowedNets(LockedTask):
    def locked_run(self, *args, **kwargs):

        allowed_nets = collect_rules(AllowedNet)
        denied_nets = collect_rules(DeniedNet)

        nets_detail = []
        for net in allowed_nets:
            nets_detail.append((net, True))

        for net in denied_nets:
            nets_detail.append((net, False))

        to_create = []
        for ip, is_allowed in nets_detail:
            rule = ip.as_rule()
            to_create.append(NetRule(
                mask=str(ip),
                is_allowed=is_allowed,
                left_begin=rule.left_begin,
                left_end=rule.left_end,
                right_begin=rule.right_begin,
                right_end=rule.right_end,
            ))

        with atomic():
            NetRule.objects.all().delete()
            NetRule.objects.bulk_create(to_create)
