from typing import Dict, Any

from django.utils.translation import get_language


class Report:
    sheet_name = ''
    sheet_outline_settings: Dict[str, Any] = {}
    freeze_panes: Dict[str, Any] = {}

    class Column(object):
        DEFAULT_COLUMN_WIDTH = 8.43

        def __init__(self, name_ru, name_en=None, column_options=None, width=DEFAULT_COLUMN_WIDTH, cell_format=None):
            self.name_ru = name_ru
            self.name_en = name_ru if name_en is None else name_en
            self.column_options = column_options or {}
            self.width = width  # default width is 8.43 according to xlsxwriter options
            self.cell_format = cell_format

        @property
        def name(self):
            return self.name_en if get_language() == 'en' else self.name_ru

    class Row(object):
        def __init__(self, cells, row_options=None, cell_format=None):
            """
            :type row_options: dict
            """
            self.cells = cells
            self.row_options = row_options or {}
            self.cell_format = cell_format

        def __repr__(self):
            return 'Row: %s' % self.cells

    class Sheet(object):
        def __init__(self, sheet_name, sheet_outline_settings, freeze_panes):
            self.sheet_name = sheet_name
            self.sheet_outline_settings = sheet_outline_settings
            self.freeze_panes = freeze_panes

    def join_data(self):
        raise NotImplementedError

    def get_rows(self):
        raise NotImplementedError

    def get_head_rows(self):
        return []

    def get_table(self):
        for row in self.get_head_rows():
            if not isinstance(row, self.Row):
                row = self.Row(row)
            yield row

        for row in self.get_rows():
            yield row

    def as_table(self):
        return [self.sheet_name, self.get_table()]

    def as_table_with_settings(self):
        return [self.Sheet(self.sheet_name, self.sheet_outline_settings, self.freeze_panes), self.get_table()]
