from bisect import bisect
from datetime import date, timedelta

from django.conf import settings
from django.db.models import Q

from staff.person.models import Staff
from staff.departments.models import Department

from ..utils import full_name_wrapper, join_staff_data


DAYS_BACK = 14
DATE_QUIT_BEFORE = date(2012, 9, 1)


class EthicReport(object):
    columns_count = 32  # 35?
    empty_line = [''] * columns_count
    captions = (
        'First Name*',
        'Middle Name',
        'Last Name*',
        'Employee ID',
        'Hire Date',
        'Address 1',
        'Address 2',
        'City',
        'Country',
        'State (US or Canada)',
        'State/Province/Region(Non US or Canadian)',
        'Zip Code/Postal Code',
        'Time Zone',
        'Email ID*',
        'Password*',
        'Phone',
        'Fax',
        'Company/Employer',
        'Job Title',
        'Supervisor Name',
        'Supervisor Email',
        'Region*',
        'Division*',
        'Department*',
        'User Language',
        'Is Author',
        'Is Realm Administrator',
        'Delete Realm',
        'New Email ID',
        'New Employee ID',
        'Custom Field 1',
        'Custom Field 2',
        'Custom Field 3',
        'Custom Field 4',
        'Action',
    )
    person_values = (
        'first_name',
        'first_name_en',
        'last_name',
        'last_name_en',
        'join_at',
        'office__city__name_en',
        'office__city__country__name_en',
        'work_email',
        'department__name_en',
        'department__lft',
        'department__rght',
        'lang_ui',
    )

    @property
    def target_staff_ds(self):
        from_date = date.today() - timedelta(days=DAYS_BACK)
        quit_condition = Q(quit_at=None) | Q(quit_at__lte=DATE_QUIT_BEFORE)
        ds = Staff.objects.filter(
            is_robot=False,
            affiliation='yandex',
            is_dismissed=False,
            join_at__gte=from_date
        ).filter(quit_condition).order_by('join_at')
        return ds.filter(lang_ui=self.lang_ui) if self.lang_ui else ds

    def __init__(self, lang_ui=None):
        self.lang_ui = lang_ui
        self.directions = self.get_directions()
        self.directions_lft = tuple(lft for lft, r_, n_ in self.directions)

    def get_directions(self):
        yandex_department_tree_id = Department.objects.get(
            id=settings.YANDEX_DEPARTMENT_ID).tree_id
        return tuple(
            dep for dep in
            Department.objects.filter(
                kind_id=settings.DIS_DIRECTION_KIND_ID,
                tree_id=yandex_department_tree_id
            ).values_list('lft', 'rght', 'name_en').order_by('lft')
        )

    def construct_line(self, pd):

        field_values = [''] * self.columns_count
        field_values[0] = pd['first_name_en'] or pd['first_name']
        field_values[2] = pd['last_name_en'] or pd['last_name']
        field_values[4] = pd['join_at'].strftime('%m/%d/%y')
        field_values[7] = pd['office__city__name_en']
        field_values[8] = pd['office__city__country__name_en']
        field_values[13] = pd['work_email']
        field_values[21] = 'No organization'
        field_values[22] = pd['department__name_en']
        field_values[23] = self.get_direction_name(
            pd['department__lft'], pd['department__rght']
        )
        # field_values[24] = pd['lang_ui']
        return [value.encode('utf-8') for value in field_values]

    def get_direction_name(self, dep_lft, dep_rght):
        index = bisect(self.directions_lft, dep_lft)
        dir_lft, dir_rght, dir_name = self.directions[index-1]
        if dir_lft < dep_lft and dep_rght < dir_rght:
            return dir_name
        return ''

    def person_lines(self):
        for person_data in self.target_staff_ds.values(*self.person_values):
            yield self.construct_line(person_data)

    def lines(self):
        yield self.captions
        yield self.empty_line
        for person_line in self.person_lines():
            yield person_line
            yield self.empty_line


def yandex_staff_gen():
    yield [
        'ФИО',
        'ФИО по данным из Оебс',
        'Логин',
        'Подразделение',
        'Цепочка подразделений',
        'ФИО руководителя',
        'Логин руководителя',
    ]

    staff = join_staff_data()

    for person in staff:
        dep = person['department']

        chief = [p for p in dep['chief_chain'] if p['login'] != person['login']]
        chief = chief and chief[0] or None

        yield [
            full_name_wrapper(person),
            person['oebs_full_name'],
            person['login'],
            dep['name'],
            ' => '.join(d['name'] for d in dep['bread_crumb']),
            full_name_wrapper(chief),
            chief['login'] if chief else '',
        ]
