from operator import and_
from itertools import groupby
from functools import reduce

from django.conf import settings
from django.db.models import Count, Q
from django.utils.functional import SimpleLazyObject

from staff.person.models import Staff
from staff.departments.models import Department, DepartmentStaff, DepartmentRoles
from staff.map.models import Office

from staff.oebs.models import Employee


HIDE_DELETED = Q(intranet_status=1)

ROLE_CHIEF = Q(role_id=DepartmentRoles.CHIEF.value)
ROLE_DEPUTY = Q(role_id=DepartmentRoles.DEPUTY.value)

HIDE_DISMISSED_STAFF = Q(staff__is_dismissed=False)
HIDE_DISMISSED = Q(is_dismissed=False)

EXCLUDE_VIRTUAL_OFFICE = Q(office=settings.VIRTUAL_OFFICE_ID)
EXCLUDE_BROKEN = Q(staff=None, department=None)


def initialize_filters():
    yandex_lft, yandex_rght, yandex_tree_id = Department.objects.filter(id=settings.YANDEX_DEPARTMENT_ID). \
        values_list('lft', 'rght', 'tree_id').get()

    outstaff_lft, outstaff_rght, outstaff_tree_id = Department.objects.filter(id=settings.OUTSTAFF_DEPARTMENT_ID). \
        values_list('lft', 'rght', 'tree_id').get()

    yandex_only_department = (Q(lft__gte=yandex_lft) & Q(rght__lte=yandex_rght) & Q(tree_id=yandex_tree_id))
    oustaff_only_department = (Q(lft__gte=outstaff_lft) & Q(rght__lte=outstaff_rght) & Q(tree_id=outstaff_tree_id))
    yandex_and_outstaff_departments = yandex_only_department | oustaff_only_department

    yandex_only = (Q(department__lft__gte=yandex_lft) & Q(department__rght__lte=yandex_rght)
                   & Q(department__tree_id=yandex_tree_id))
    yandex_and_outstaff_only = (
        (yandex_only) | (
            Q(department__lft__gte=outstaff_lft)
            & Q(department__rght__lte=outstaff_rght)
            & Q(department__tree_id=outstaff_tree_id)
        )
    )
    yandex_only_from_staff = (
        Q(staff__department__lft__gte=yandex_lft)
        & Q(staff__department__rght__lte=yandex_rght)
        & Q(staff__department__tree_id=yandex_tree_id)
    )

    return {
        "YANDEX_ONLY": yandex_only,
        "YANDEX_ONLY_DEPARTMENT": yandex_only_department,
        "YANDEX_AND_OUTSTAFF_ONLY": yandex_and_outstaff_only,
        "YANDEX_AND_OUTSTAFF_DEPARTMENTS": yandex_and_outstaff_departments,
        "YANDEX_ONLY_FROM_STAFF": yandex_only_from_staff
    }


filters = SimpleLazyObject(initialize_filters)


def get_departments(*filter_options):
    fields = ['id', 'url', 'name', 'name_en', 'parent_id']
    filter_options = reduce(and_, filter_options, Q())

    departments = (
        Department.objects
        .filter(filter_options)
        .values(*fields)
        .order_by('lft')
    )
    return departments


def get_value_streams(*filter_options):
    fields = ['id', 'url', 'name', 'name_en', 'parent_id']
    filter_options = reduce(and_, filter_options, Q())

    departments = (
        Department.valuestreams
        .filter(filter_options)
        .values(*fields)
        .order_by('lft')
    )
    return departments


def get_offices():
    offices = (
        Office.objects
        .values(
            'id',
            'name',
            'name_en',
        )
    )
    return offices


def get_chiefs(*filter_options):
    from staff.reports.utils import person_wrapper
    fields = [
        'department',
        'staff__login',
        'staff__first_name',
        'staff__middle_name',
        'staff__last_name',
    ]
    filter_options = reduce(and_, filter_options, ROLE_CHIEF)

    chiefs = (
        DepartmentStaff.objects
        .exclude(EXCLUDE_BROKEN)
        .filter(filter_options)
        .values(*fields)
    )

    chiefs = {ds['department']: person_wrapper(ds) for ds in chiefs}
    return chiefs


def get_deputies(*filter_options):
    from staff.reports.utils import person_wrapper
    fields = [
        'department',
        'staff__login',
        'staff__first_name',
        'staff__middle_name',
        'staff__last_name',
    ]
    filter_options = reduce(and_, filter_options, ROLE_DEPUTY)

    deputies = (
        DepartmentStaff.objects
        .exclude(EXCLUDE_BROKEN)
        .filter(filter_options)
        .values(*fields)
        .order_by('department__url')
    )

    deputies = {
        dep_id: [person_wrapper(dep) for dep in dep_deputies]
        for dep_id, dep_deputies in groupby(deputies, key=lambda ds: ds['department'])
    }

    return deputies


def get_persons(*filter_options):
    fields = ['id', 'department', 'login', 'first_name', 'middle_name', 'last_name']
    filter_options = reduce(and_, filter_options, Q())

    return (
        Staff.objects
        .filter(filter_options)
        .values(*fields)
    )


def get_persons_count():
    persons_count = (
        get_persons(HIDE_DISMISSED, filters['YANDEX_AND_OUTSTAFF_ONLY'])
        .values('department')
        .annotate(count=Count('id'))
    )
    persons_count = {pc['department']: pc['count'] for pc in persons_count}
    return persons_count


def get_oebs_full_names():
    qs = (
        Employee.objects
        .exclude(dis_staff=None)
        .values_list('dis_staff_id', 'last_name', 'first_name', 'middle_names')
    )
    return {
        staff_id: '{} {} {}'.format(last_name, first_name, middle_name)
        for staff_id, last_name, first_name, middle_name in qs
    }
