from collections import OrderedDict

from .sources import (
    get_chiefs,
    get_deputies,
    get_persons,
    get_persons_count,
    get_departments,
    get_oebs_full_names,
    filters,

    HIDE_DELETED,
    HIDE_DISMISSED,
    EXCLUDE_VIRTUAL_OFFICE,
    HIDE_DISMISSED_STAFF,
)


class ChiefChainGetter(object):
    def __init__(self, chiefs):
        self.chiefs = chiefs
        self._cache = {}

    def get_chief_chain(self, dep):
        chain = [self.chiefs.get(dep['id'])]
        if dep['parent'] is not None:
            chain.extend(self.get_chief_chain(dep['parent']))
        return [chief for chief in chain if chief]


def populate_departments(departments_qs, chiefs=None):
    chiefs = chiefs or get_chiefs()
    departments = OrderedDict()
    chiefs_getter = ChiefChainGetter(chiefs)

    for dep in departments_qs:
        dep = dict(dep)
        parent_dep = departments.get(dep['parent_id'])
        dep['parent'] = parent_dep
        # chief
        direct_chief = chiefs.get(dep['id'])
        if direct_chief:
            dep['chief'] = direct_chief
        elif parent_dep:
            dep['chief'] = parent_dep.get('chief')
        dep['chief_chain'] = chiefs_getter.get_chief_chain(dep)

        # bread_crumb
        parent_bread_crumb = parent_dep['bread_crumb'] if parent_dep else []
        dep['bread_crumb'] = parent_bread_crumb + [dep]

        departments[dep['id']] = dep

    return departments


def person_wrapper(person):
    if person is None:
        return {}
    return {
        'login': person['staff__login'],
        'first_name': person['staff__first_name'],
        'middle_name': person['staff__middle_name'],
        'last_name': person['staff__last_name'],
    }


def full_name_wrapper(person):
    if person is None:
        return ''
    return ' '.join(
        person[field] for field in ['first_name', 'middle_name', 'last_name']
    )


def join_staff_data():
    departments_qs = get_departments(HIDE_DELETED, filters['YANDEX_ONLY_DEPARTMENT'])
    oebs_full_names = get_oebs_full_names()

    departments = populate_departments(departments_qs)

    target_persons = (
        get_persons(HIDE_DISMISSED, filters['YANDEX_ONLY'])
        .exclude(EXCLUDE_VIRTUAL_OFFICE)
        .order_by('id')
    )
    for person in target_persons:
        person['department'] = departments[person['department']]
        person['oebs_full_name'] = oebs_full_names.get(person.pop('id'), '')
        yield person


def join_departments_data():
    chiefs = get_chiefs(HIDE_DISMISSED_STAFF)
    persons_count = get_persons_count()
    deputies = get_deputies(HIDE_DISMISSED_STAFF, filters['YANDEX_AND_OUTSTAFF_ONLY'])
    departments_qs = get_departments(HIDE_DELETED, filters['YANDEX_AND_OUTSTAFF_DEPARTMENTS'])
    departments = populate_departments(departments_qs, chiefs=chiefs)

    for dep_id, dep in departments.items():
        # persons_count
        dep['direct_chief'] = chiefs.get(dep_id)
        dep['direct_persons_count'] = persons_count.get(dep_id, 0)
        dep['persons_count'] = 0
        for ancestor in dep['bread_crumb']:
            ancestor['persons_count'] += dep['direct_persons_count']

        # deputies
        dep['deputies'] = deputies.get(dep['id'], [])
    return departments
