from operator import attrgetter

from django.utils.translation import get_language
from staff.rfid.exceptions import (
    DoesNotExist,
    MultipleObjectsReturned,
    RfidModifyReadOnlyFieldError,
)


class ListController(object):
    _requested_fields = None

    def __init__(self, ctl_params=None):
        self.ctl_params = ctl_params or {}
        self.qs = self.get_query_set()

    @property
    def controllers(self):
        raise NotImplementedError

    def get_controller(self, obj):
        for ctl in self.controllers:
            if ctl.condition(obj):
                return ctl

    def wrap_controller(self, obj):
        ctl = self.get_controller(obj)
        instance = ctl(obj, self, **self.ctl_params)
        if isinstance(obj, dict):
            return instance.as_dict(*self._requested_fields)
        return instance

    def get_query_set(self):
        return self.model.objects.all()

    def get(self, **kwargs):
        try:
            obj = self.qs.get(**kwargs)
        except self.model.DoesNotExist as e:
            raise DoesNotExist(e)
        except self.model.MultipleObjectsReturned as e:
            raise MultipleObjectsReturned(e)

        return self.wrap_controller(obj)

    def first(self):
        result = self.qs.first()

        if not result:
            return None

        return self.wrap_controller(result)

    def create_new_obj(self, obj_data):
        return self.model()

    def create(self, **kwargs):
        ctl = self.get_controller(kwargs)
        obj = self.create_new_obj(kwargs)
        return ctl(obj, **self.ctl_params).create(**kwargs)

    def all(self):
        self.qs = self.qs.all()
        return self

    def order_by(self, *args):
        self.qs = self.qs.order_by(*args)
        return self

    def filter(self, *args, **kwargs):
        self.qs = self.qs.filter(*args, **kwargs)
        return self

    def exclude(self, *args, **kwargs):
        self.qs = self.qs.exclude(*args, **kwargs)
        return self

    def values(self, *fields):
        assert fields, 'You must provide some fields!'
        if self._requested_fields is None:
            self._requested_fields = list(fields)
        self.qs = self.qs.values(*fields)
        return self

    def _iterator(self, qs):
        for obj in qs:
            yield self.wrap_controller(obj)

    def __iter__(self):
        return self._iterator(self.qs)

    def __getitem__(self, key):
        if isinstance(key, slice):
            return self._iterator(self.qs[key])
        else:
            return self.wrap_controller(self.qs[key])


class ReadOnlyField(object):
    _num = 0

    @classmethod
    def get_num(cls):
        ReadOnlyField._num += 1
        return cls._num

    def __init__(self, field_name='', i18n=False):
        self.num = self.get_num()

        self.field_name = field_name
        self.i18n = i18n

    def __get__(self, instance, owner):
        return instance._getattr(self.field_name, self.i18n)

    def __set__(self, instance, value):
        raise RfidModifyReadOnlyFieldError(self.field_name)


class EditableField(ReadOnlyField):

    def __set__(self, instance, value):
        setattr(instance.obj, self.field_name, value)


class NoneField(ReadOnlyField):

    def __get__(self, instance, owner):
        return None


class BaseController(object):

    _controller_fields = None
    _controller_fields_class = ReadOnlyField

    def __init__(self, obj, list_controller=None):
        self.list_controller = list_controller
        self.obj = obj

    @classmethod
    def condition(cls, obj):
        return True

    @classmethod
    def ordered_controller_fields(cls):
        if cls._controller_fields is None:
            cls._controller_fields = []
            for _class in cls.mro():
                ro_fields = [
                    (field.num, name) for name, field in _class.__dict__.items()
                    if isinstance(field, cls._controller_fields_class)
                ]
                for num, field_name in sorted(ro_fields):
                    if field_name not in cls._controller_fields:
                        cls._controller_fields.append(field_name)

        return cls._controller_fields

    def _getattr(self, field, i18n=False):
        if isinstance(self.obj, dict):
            if i18n and field + '_en' in self.obj:
                field = field if 'ru' in get_language() else field + '_en'
            return self.obj[field]

        obj = self.obj

        if '__' in field:
            route, field = field.rsplit('__', 1)
            route = route.replace('__', '.')
            getter = attrgetter(route)
            try:
                obj = getter(obj)
            except AttributeError:
                return None

        if i18n and hasattr(obj, field + '_en'):
            field = field if 'ru' in get_language() else field + '_en'

        if obj is None:
            return None
        else:
            return getattr(obj, field)

    def create(self, **kwargs):
        for attr_name in self.ordered_controller_fields():
            if attr_name in kwargs:
                setattr(self, attr_name, kwargs.pop(attr_name))

        for attr, value in kwargs.items():
            setattr(self, attr, value)

        self.save()
        return self

    def as_dict(self, *fields):
        result = dict(
            (field, getattr(self, field))
            for field in fields
            if hasattr(self, field)
        )
        return result

    def save(self):
        self.obj.save()
