from django.db.models import Q

from staff.preprofile.repository import Repository as PreprofilesRepository
from staff.rfid.exceptions import DoesNotExist, RfidCandidateBadgeAlreadyExist
from staff.rfid.controllers.base import BaseController, ReadOnlyField
from staff.rfid.controllers.badge import Badges
from staff.rfid.constants import OWNER


class PreprofileIDField(ReadOnlyField):

    def __init__(self, prefix=''):
        self.prefix = prefix

    def __get__(self, instance, owner):
        return instance._getattr('id')


class NewCandidate(BaseController):

    preprofile_id = PreprofileIDField('id')
    join_at = ReadOnlyField('join_at')
    office_id = ReadOnlyField('office_id')
    first_name = ReadOnlyField('first_name')
    last_name = ReadOnlyField('last_name')
    first_name_en = ReadOnlyField('first_name_en')
    last_name_en = ReadOnlyField('last_name_en')
    login = ReadOnlyField('login')
    recruiter_id = ReadOnlyField('recruiter_id')

    def to_candidate(self, code=None):
        try:
            Badges().get(preprofile_id=self.preprofile_id)
        except DoesNotExist:
            pass
        else:
            raise RfidCandidateBadgeAlreadyExist

        candidate_data = {
            'first_name': self.first_name,
            'first_name_en': self.first_name_en,
            'last_name': self.last_name,
            'last_name_en': self.last_name_en,
            'preprofile_id': self.preprofile_id,
            'office_id': self.office_id,
            'login': self.login,
        }
        if code:
            candidate_data['code'] = code
        candidate = Badges().create_candidate(**candidate_data)
        return candidate


class NewCandidates(object):
    _requested_fields = None

    def __init__(self, observer):
        self.query = PreprofilesRepository(observer).preprofiles_qs()

    controller = NewCandidate

    def wrap_controller(self, obj):
        instance = self.controller(obj, self)
        if self._requested_fields:
            return instance.as_dict(*self._requested_fields)
        return instance

    def get(self, **kwargs):
        result = self.query.get(**kwargs)
        return self.controller(result, self)

    def all(self):
        return self

    def actual(self):
        existing_ids = []
        for item in Badges().filter(owner=OWNER.CANDIDATE).values('preprofile_id'):
            if item['preprofile_id']:
                existing_ids.append(item['preprofile_id'])
        self.exclude(id__in=existing_ids)
        return self

    def search(self, search_str):
        if not search_str:
            return self

        search_fields = [
            'id',
            'login',
            'first_name',
            'last_name',
            'first_name_en',
            'last_name_en',
        ]
        for bit in search_str.split():
            or_queries = None
            for search_field in search_fields:
                q = Q(**{search_field + '__icontains': bit})
                if or_queries is None:
                    or_queries = q
                else:
                    or_queries |= q
            self.query = self.query.filter(or_queries)

        return self

    def order_by(self, *args):
        if args:
            self.query = self.query.order_by(*args)
        return self

    def filter(self, **kwargs):
        if kwargs:
            # kwargs = self.replace_filter_fields(kwargs)
            self.query = self.query.filter(**kwargs)
        return self

    def exclude(self, **kwargs):
        if kwargs:
            # kwargs = self.replace_filter_fields(kwargs)
            self.query = self.query.exclude(**kwargs)
        return self

    def values(self, *fields):
        assert fields, 'You must provide some fields!'
        self._requested_fields = list(fields)
        q_fields = set(fields)
        if 'preprofile_id' in fields:
            q_fields.add('id')
            q_fields.discard('preprofile_id')

        if any(f in q_fields for f in [
            'first_name',
            'first_name_en'
        ]):
            q_fields.add('first_name')
            q_fields.add('first_name_en')

        if any(f in q_fields for f in [
            'last_name',
            'last_name_en'
        ]):
            q_fields.add('last_name')
            q_fields.add('last_name_en')

        self.query = self.query.values(*q_fields)
        return self

    def _iterator(self, item_key=None):
        objects = self.query
        if item_key:
            objects = objects[item_key]

        for obj in objects:
            yield self.wrap_controller(obj)

    def __iter__(self):
        return self._iterator()

    def __getitem__(self, key):
        if isinstance(key, slice):
            return self._iterator(item_key=key)
        else:
            try:
                return next(self._iterator())
            except StopIteration:
                raise KeyError
