from staff.rfid.models import Rfid, Badge
from .base import ListController, BaseController, EditableField, ReadOnlyField
from staff.rfid.constants import CODE_STATUS
from staff.rfid.exceptions import (
    DoesNotExist,
    RfidCodeEmptyError,
    RfidCodeBusyError,
    FieldIsRequired,
    RfidCodeRewriteError,
    RfidCodeDeleteError,
)


class ReserveCodeField(EditableField):

    def __init__(self, **kwargs):
        super(ReserveCodeField, self).__init__('code', **kwargs)

    def __set__(self, instance, value):
        if not value:
            raise RfidCodeEmptyError

        current_code = instance._getattr(self.field_name)
        if current_code and current_code != value:
            raise RfidCodeRewriteError

        instance.obj.code = value


class Reserve(BaseController):

    code = ReserveCodeField()
    is_active = ReadOnlyField('is_active')

    def delete(self):
        try:
            if self.obj.badge:
                raise RfidCodeDeleteError
        except (self.obj.DoesNotExist, Badge.DoesNotExist):
            pass
        self.obj.is_active = False
        self.obj.save()

    def save(self):
        if not self.code:
            raise FieldIsRequired('code')
        super(Reserve, self).save()


class Reserves(ListController):
    model = Rfid
    controllers = Reserve,

    def get_query_set(self):
        return (
            super(Reserves, self)
            .get_query_set()
            .select_related('badge')
            .filter(badge=None)
        )

    def create_new_obj(self, obj_data):
        code = obj_data.get('code')

        if code is None:
            raise RfidCodeEmptyError

        code_state, rfid = self.code_status_and_instance(code)

        if code_state == CODE_STATUS.NEW:
            return self.model()
        elif code_state in (CODE_STATUS.IN_RESERVE, CODE_STATUS.USED):
            raise RfidCodeBusyError
        elif code_state == CODE_STATUS.DELETED:
            rfid.is_active = True
            return rfid

    def get_or_create(self, code, **kwargs):
        try:
            return self.get(code=code), False
        except DoesNotExist:
            return self.create(code=code, **kwargs), True

    @classmethod
    def code_status_and_instance(cls, code):
        try:
            rfid = cls.model.objects.select_related('badge').get(code=code)
        except cls.model.DoesNotExist:
            return CODE_STATUS.NEW, None

        try:
            rfid.badge
        except Badge.DoesNotExist:
            if rfid.is_active:
                return CODE_STATUS.IN_RESERVE, rfid
            else:
                return CODE_STATUS.DELETED, rfid
        else:
            return CODE_STATUS.USED, rfid


def code_status(code):
    return Reserves.code_status_and_instance(code)[0]
