from typing import Dict

from django.db import models

from staff.users.models import User
from staff.person.models import Staff
from staff.rfid.constants import OWNER, STATE, REASON
from staff.rfid.models.base import BaseRfidModel
from staff.rfid.models.rfid import Rfid
from staff.preprofile.models import Preprofile


class Badge(BaseRfidModel):
    class Meta:
        permissions = (
            ('list_reserve', 'API:Can see reserve keys'),
            ('edit_reserve', 'API:Can edit reserve keys'),
            ('list_candidates', 'API:Can see candidates'),
            ('edit_candidates', 'API:Can edit candidates'),
            ('list_anonymous', 'API:Can see anonymous'),
            ('edit_anonymous', 'API:Can edit anonymous'),
            ('list_employees', 'API:Can see employees'),
            ('edit_employees', 'API:Can edit employees'),
            ('use_list_handle', 'Robot-API:Can use list handle'),
            ('use_key_handle', 'Robot-API:Can use key handle'),
            ('can_export_payers', 'API: Can export payers'),

            ('change_anonym_food', 'Can change anonym food permission'),
            ('robot_with_rfid_api_access', 'Permission for robots group, allows access to rfid api'),
        )

    rfid = models.OneToOneField(Rfid, null=True, blank=True)
    owner = models.CharField(choices=OWNER, max_length=20)
    state = models.CharField(choices=STATE, max_length=20)
    reason = models.CharField(
        choices=REASON, max_length=20, null=True, blank=True,
    )
    changed_by = models.ForeignKey(Staff, null=True, blank=True)

    # EDITABLE
    # Employee
    person = models.ForeignKey(Staff, null=True, blank=True, related_name='badges')

    # Anon
    contractor = models.ForeignKey('ContractorFirm', null=True, blank=True)
    anonym_food_allowed = models.BooleanField(default=False)

    # Anon, Candidate
    first_name = models.CharField(max_length=256, null=True, blank=True)
    first_name_en = models.CharField(max_length=256, null=True, blank=True)
    last_name = models.CharField(max_length=256, null=True, blank=True)
    last_name_en = models.CharField(max_length=256, null=True, blank=True)

    # Candidate
    preprofile = models.ForeignKey(Preprofile, null=True, blank=True, unique=True)
    login = models.CharField(max_length=4096, null=True, blank=True)
    photo = models.CharField(max_length=4096, null=True, blank=True)
    position = models.TextField(null=True, blank=True)
    join_at = models.DateField(null=True, blank=True)
    office_id = models.IntegerField(null=True, blank=True)

    def __str__(self):
        if self.person:
            name = self.person.login
        else:
            name = self.first_name or self.first_name_en or ''
        return '<Badge: {0.owner} {1}>'.format(self, name)


def get_permissions(user: User) -> Dict[str, bool]:
    """ Return status (True/False) for all custom rfid permissions """

    def get_key(perm_name):
        return 'rfid.{0}'.format(perm_name)

    return {
        get_key(perm_name): user.has_perm(get_key(perm_name))
        for perm_name, perm_desc in Badge._meta.permissions
    }


class BadgeChange(BaseRfidModel):
    badge_id = models.IntegerField()
    code = models.CharField(max_length=128)
    state = models.CharField(max_length=32, choices=STATE)
    person = models.ForeignKey(Staff, null=True, blank=True)
    is_sent = models.BooleanField(default=False)
    is_sent_to_hd = models.BooleanField(default=False)
    last_update = models.DateTimeField(null=True)
