from typing import Tuple, List

import logging

from datetime import datetime, timedelta

from staff.lib import waffle

from django.conf import settings

from django.db.models import Q

from staff.celery_app import app
from staff.lib.tasks import LockedTask

from staff.lib import requests, tvm2
from staff.lib.db import atomic
from staff.lib.utils.qs_values import extract_related

from staff.person.models import Staff
from staff.rfid.controllers import Badges
from staff.rfid.notifications import send_badges_mail
from staff.rfid.constants import OWNER, STATE
from staff.rfid.models import BadgeChange
from staff.rfid.utils import PayerBadge, PayersReport

logger = logging.getLogger('staff.rfid.tasks')


def fire_all(person_ids=None):
    badges = (
        Badges()
        .filter(owner=OWNER.EMPLOYEE, person__is_dismissed=True)
        .exclude(state__in=[
            STATE.LOST, STATE.INACTIVE, STATE.DISCHARGED, STATE.BLOCKED
        ])
    )

    if person_ids:
        badges.filter(person__in=person_ids)

    for badge in badges:
        badge.dismiss()


def hire_all(person_ids=None):
    logins = Badges().filter(owner=OWNER.CANDIDATE).values('login')
    logins = (l['login'] for l in logins)

    new_employees = Staff.objects.filter(login__in=logins, is_dismissed=False).exclude(badges__state=STATE.ACTIVE)
    if person_ids:
        new_employees = new_employees.filter(id__in=person_ids)
    new_employees = {p.login: p for p in new_employees}

    badges = Badges().filter(owner=OWNER.CANDIDATE, login__in=new_employees.keys())

    for badge in badges:
        person = new_employees[badge.login]
        try:
            badge.to_employee(person)
        except Exception:
            logger.exception('When transfer candidate to employee login %s, code %s', badge.login, badge.code)


@app.task(ignore_result=True)
class BadgesRecreated(LockedTask):
    """
    Send Email about recent recreated badges
    """
    def locked_run(self):
        send_badges_mail()


@app.task(ignore_result=True)
class HirePeople(LockedTask):
    """
    Change status to EMPLOYEE for all candidates that have been hired now
    """
    def locked_run(self, person_ids=None):
        hire_all(person_ids)


@app.task(ignore_result=True)
class FirePeople(LockedTask):
    """
    Change status to 'discharged' for people who are fired
    """
    def locked_run(self, person_ids=None):
        fire_all(person_ids)


@app.task(ignore_result=True)
class SyncAll(LockedTask):

    def locked_run(self):
        fire_all()
        hire_all()


@app.task(ignore_result=True)
class SyncBadgeChanges(LockedTask):
    @atomic
    def locked_run(self):
        badges = list(
            Badges()
            .exclude(state=STATE.NOCODE)
            .filter(owner__in=[OWNER.ANONYM, OWNER.EMPLOYEE])
            .filter(updated_at__gte=datetime.utcnow() - timedelta(hours=1))
            .values('id', 'code', 'person', 'state', 'updated_at', 'anonym_food_allowed', 'owner')
        )

        query_to_find_repetition = ~Q()
        for badge in badges:
            query_to_find_repetition |= Q(badge_id=badge['id'], last_update=badge['updated_at'])

        updated_badges = BadgeChange.objects.filter(query_to_find_repetition).values_list('badge_id', flat=True)
        badges_to_update = set([badge['id'] for badge in badges]) - set(updated_badges)

        (
            BadgeChange.objects
            .filter(badge_id__in=badges_to_update)
            .filter(Q(is_sent=False) | Q(is_sent_to_hd=False))
            .delete()
        )

        def get_state(badge):
            if badge['owner'] == OWNER.ANONYM and not badge['anonym_food_allowed']:
                return STATE.INACTIVE
            return badge['state']

        # sorry

        changes = [
            BadgeChange(
                badge_id=badge['id'],
                code=badge['code'],
                state=get_state(badge),
                person_id=badge['person'],
                last_update=badge['updated_at'],
                is_sent=False,
                is_sent_to_hd=True,
            ) for badge in badges if badge['id'] in badges_to_update
        ] + [
            BadgeChange(
                badge_id=badge['id'],
                code=badge['code'],
                state=badge['state'],
                person_id=badge['person'],
                last_update=badge['updated_at'],
                is_sent=True,
                is_sent_to_hd=False,
            ) for badge in badges if badge['id'] in badges_to_update
        ]

        BadgeChange.objects.bulk_create(changes)


def prepare_badge_changes_report(filter_: Q, limit: int or None = None) -> Tuple[List, List]:
    fields = [
        'id', 'badge_id', 'code', 'state',
        'person__first_name', 'person__last_name', 'person__middle_name', 'person__login'
    ]

    badge_changes = BadgeChange.objects.filter(filter_).values(*fields)
    if limit is not None:
        badge_changes = badge_changes[:limit]
    changes = list(badge_changes)

    if not changes:
        return [], []

    changes_ids = []
    for change in changes:
        change.update(extract_related(change, 'person'))
        changes_ids.append(change.pop('id'))
        change['id'] = change.pop('badge_id')
        change['owner'] = OWNER.EMPLOYEE if change['login'] else OWNER.ANONYM

    payer_badges = [PayerBadge(**change) for change in changes]
    return PayersReport(payer_badges).create_report(), changes_ids


@app.task(ignore_result=True)
class ExportBadgeChanges(LockedTask):
    def locked_run(self):
        self.send_changes_to_oebs()
        if waffle.switch_is_active("enable_badge_changes_for_helpdesk"):
            self.send_changes_to_helpdesk()

    def send_changes_to_oebs(self):
        report, changes = prepare_badge_changes_report(Q(is_sent=False))

        try:
            request = requests.post(
                url=f'https://{settings.BADGEPAY_HOST}/pg/integration/payers',
                json=report,
                headers={tvm2.TVM_SERVICE_TICKET_HEADER: tvm2.get_tvm_ticket_by_deploy('badgepay')},
                timeout=(1, 2, 5),
                no_log_json=True,
            )
        except Exception as e:
            logger.exception('Fail to push badge changes to oebs %s', str(e))
            return

        if request.status_code == 200:
            BadgeChange.objects.filter(id__in=changes).update(is_sent=True)

    def send_changes_to_helpdesk(self):
        report, changes = prepare_badge_changes_report(Q(is_sent_to_hd=False), limit=10)

        try:
            request = requests.post(
                url=f'https://{settings.HELPDESK_HOST}/api/external/lenel/webhooks/staff',
                json=report,
                headers={'Authorization': f'OAuth {settings.ROBOT_STAFF_OAUTH_TOKEN}'},
                timeout=(120, 240),
                no_log_json=True,
            )
        except Exception as e:
            logger.exception('Fail to push badge changes to helpdesk %s', str(e))
            return

        if request.status_code == 200:
            BadgeChange.objects.filter(id__in=changes).update(is_sent_to_hd=True)
