from typing import List, DefaultDict, Union, Optional

import attr

from collections import defaultdict
from phonenumbers import parse, format_number, PhoneNumberFormat, NumberParseException

from staff.person.models import StaffPhone, PHONE_PROTOCOLS, PHONE_KIND
from staff.rfid.constants import OWNER, STATE


def owner_converter(owner):
    return 'Employee' if owner == OWNER.EMPLOYEE else 'Guest'


def state_converter(state):
    return 'active' if state == STATE.ACTIVE else 'inactive'


@attr.s
class PayerBadge:
    id: Optional[int] = attr.ib(converter=attr.converters.optional(int))
    code: Optional[int] = attr.ib(converter=attr.converters.optional(int))
    owner = attr.ib(converter=attr.converters.optional(owner_converter))  # type: ignore
    state = attr.ib(converter=attr.converters.optional(state_converter))  # type: ignore
    login = attr.ib(default=None)
    first_name = attr.ib(default=None)
    last_name = attr.ib(default=None)
    middle_name = attr.ib(default=None)


class PayersReport(object):
    def __init__(self, badges):  # type: (List[PayerBadge]) -> None
        self._phones = None
        self.person_badges = defaultdict(list)  # type: DefaultDict[Union[str, int], list]
        for badge in badges:
            self.person_badges[badge.login if badge.login else badge.id].append(badge)

    def __iter__(self):
        for badges in self.person_badges.values():
            yield self.create_payer(badges)

    def create_report(self):
        return [payer for payer in self]

    @property
    def phones(self):
        if not self._phones:
            self._phones = self.get_mobile_phones()
        return self._phones

    def _get_all_and_sms_mobile_phones(self):
        return (
            StaffPhone.objects
            .filter(
                staff__login__in=self.person_badges.keys(),
                protocol__in=[PHONE_PROTOCOLS.ALL, PHONE_PROTOCOLS.SMS],
                kind=PHONE_KIND.COMMON,
                intranet_status=1,
            )
            .order_by('protocol', '-position')  # Телефоны с типом SMS будут в конце
            .values_list('staff__login', 'number')
        )

    def get_mobile_phones(self):
        phones = defaultdict(lambda: None)

        for login, phone_number in self._get_all_and_sms_mobile_phones():
            try:
                phone = format_number(parse(phone_number), PhoneNumberFormat.E164)
                phones[login] = phone
            except NumberParseException:
                phones.setdefault(login, None)

        return phones

    def create_payer(self, badges):
        assert badges
        fields = {
            'Guest': ['type', 'id', 'badges'],
            'Employee': ['type', 'login', 'firstName', 'lastName', 'middleName', 'phone', 'badges'],
        }
        payer = {
            'id': badges[0].id,
            'type': badges[0].owner,
            'login': badges[0].login,
            'firstName': badges[0].first_name,
            'lastName': badges[0].last_name,
            'middleName': badges[0].middle_name,
            'phone': self.phones[badges[0].login],
            'badges': [
                {
                    'rfid': badge.code,
                    'status': badge.state,
                }
                for badge in badges
            ],
        }
        return {
            field: payer[field]
            for field in fields[badges[0].owner]
        }
