import re
import logging
import itertools
from datetime import date, timedelta
import trafaret as t

from django.db.models import Q
from django.http import HttpResponse

from staff.person_avatar.models import AvatarMetadata
from staff.person_avatar.storage import AvatarBadRequestError
from staff.person.models import Staff
from staff.preprofile.models import Preprofile
from staff.map.models import Office
from staff.lib import apilib
from staff.lib.apilib import InputParam
from staff.lib.utils.ordered_choices import OrderedChoices
from staff.lib.utils.qs_values import localize
from staff.lib.decorators import responding_json, make_json_response, auth_by_tvm_only, available_for_external

from staff.rfid.utils import PayersReport, PayerBadge
from staff.rfid.constants import OWNER, STATE
from staff.rfid.controllers import Contractors, Reserves, Badges, NewCandidates
from staff.rfid.controllers.reserve import code_status
from staff.rfid.exceptions import (
    DoesNotExist,
    RfidCodeError,
    RfidCodeRewriteError,
    RfidPersonHasBadgeError,
    RfidUnavailableStateTransition,
    RfidCandidateBadgeAlreadyExist,
)
from staff.rfid.models import get_permissions
from staff.rfid.validators import (
    ChoiceValidator,
    StaffLoginValidator,
    OrgValidator,
    CodeValidator,
    file_validator,
)


logger = logging.getLogger(__name__)


RfidApi = apilib.Collection(
    doc_url='docs',
    help='Rfid 2.0 api collection',
    formats=['json']
)


# Magic 'all' option for filter parameters that means 'no filter'
magic_all = t.Atom('all') >> (lambda x: None)


OWNER_EDIT_PERMS = {
    OWNER.ANONYM: 'rfid.edit_anonymous',
    OWNER.EMPLOYEE: 'rfid.edit_employees',
    OWNER.CANDIDATE: 'rfid.edit_candidates'
}

OWNER_VIEW_PERMS = {
    OWNER.ANONYM: 'rfid.list_anonymous',
    OWNER.EMPLOYEE: 'rfid.list_employees',
    OWNER.CANDIDATE: 'rfid.list_candidates'
}


@RfidApi.register
class Organisations(apilib.ApiMethod):
    url = r'^orgs.{format}/?$'
    help = 'List contractor organisations'

    def GET(self):
        res = Contractors().values_active('id', 'name', 'desc')
        return self.OK('Listing organizations', data=list(res))


@RfidApi.register
class Meta(apilib.ApiMethod):
    help = 'Basic context needed to render the page'
    url = r'^meta.{format}/?$'

    def GET(self):
        user = self.request.user
        data = dict(
            permissions=get_permissions(user),
            ownertypes=dict(OWNER),
            statuses=dict(STATE),
            offices=dict(
                (off.pk, off.i_name)
                for off in (
                    Office.objects
                    .filter(intranet_status=1)
                    .order_by('position')
                )
            )
        )
        return self.OK('Meta and permissions for %s' % user,
                       data=data)


@RfidApi.register
class CheckCode(apilib.ApiMethod):
    url = r'^check_code.{format}/?$'
    help = 'Return key status (new, used, in_reserve)'

    code = InputParam('Code to check', validator=CodeValidator, required=True)

    def GET(self):
        status = code_status(self.p.code)
        return self.OK('OK', data={'status': status})


@RfidApi.register
class ListReserves(apilib.ApiMethod):
    url = r'^reserves/list.{format}/?$'
    help = 'List reserve keys'
    require_permissions = ['rfid.list_reserve']

    limit = InputParam('Max results', validator=t.Int(gt=0))
    offset = InputParam('Results offset', validator=t.Int(gte=0))

    def GET(self):
        limit = self.inputs.limit
        offset = self.inputs.offset or 0

        if limit:
            reserves = list(
                Reserves()
                .filter(is_active=True)
                .values('code')
                [offset:offset + limit + 1]
            )
            more = len(reserves) > limit
            if more:
                reserves = reserves[:-1]

        else:
            reserves = list(Reserves().filter(is_active=True).values('code'))
            more = False

        count = len(reserves)
        data = {
            'reserves': reserves,
            'count': count,
            'more': more,
        }
        return self.OK('Listing reserve keys for ya.', data=data)


class CodeErrorMixin(object):
    def _make_code_error_response(self, error, code, desc):
        return self.CUSTOM_RESPONSE({
            'errors': [{'code': code, 'desc': desc}],
            'content': {
                'code': [
                    {
                        'code': error.__class__.__name__,
                        'desc': error.__doc__,
                        'status': code_status(self.p.code),
                    }
                ]
            }
        })


@RfidApi.register
class CreateReserve(apilib.ApiMethod, CodeErrorMixin):
    help = 'Add a key to reserves'
    url = r'^reserves/add.{format}/?$'
    require_permissions = ['rfid.edit_reserve']

    code = InputParam('Code to add', validator=CodeValidator, required=True)

    def POST(self):
        try:
            Reserves().create(code=self.p.code)
        except RfidCodeError as error:
            return self._make_code_error_response(
                error, 'create_reserve_error', 'Failed to add code to reserve'
            )
        else:
            return self.OK()


@RfidApi.register
class DeleteReserve(apilib.ApiMethod, CodeErrorMixin):
    help = 'Delete a reserve key'
    url = r'^reserves/delete.{format}/?$'
    require_permissions = ['rfid.edit_reserve']

    code = InputParam('Code to delete', validator=CodeValidator, required=True)

    def POST(self):
        try:
            Reserves().get(code=self.inputs.code, is_active=True).delete()
        except DoesNotExist:
            return self.NOT_FOUND()
        except RfidCodeError as error:
            return self._make_code_error_response(
                error,
                'delete_reserve_error',
                'Failed to delete code from reserve'
            )
        else:
            return self.OK('Code deleted')


@RfidApi.register
class ListBadges(apilib.ApiMethod):
    url = r'^badges/list.{format}/?$'
    help = 'List all non-reserve badges'

    limit = InputParam('Max results', validator=t.Int(gt=0, lte=100),
                       required=True)
    offset = InputParam('Results offset', validator=t.Int(gte=0))
    owner = InputParam('Owner type',
                       validator=ChoiceValidator(OWNER) | magic_all)
    status = InputParam('Status',
                        validator=ChoiceValidator(STATE) | magic_all)

    search = InputParam('Search string', validator=t.String)

    def allowed_owners(self):
        return set(
            owner for (owner, perm) in OWNER_VIEW_PERMS.items()
            if self.request.user.has_perm(perm)
        )

    def GET(self):
        limit = self.inputs.limit
        offset = self.inputs.offset or 0
        owners = self.allowed_owners()

        lookup = {'owner__in': owners}
        if self.p.owner:
            lookup.update(owner=self.p.owner)
        if self.p.status:
            lookup.update(state=self.p.status)
        search = self.p.search
        data = list(
            Badges()
            .values(
                'code',
                'owner',
                'full_name',
                'login',
                'state',
                'contractor_name',
                'id',
            )
            .search(search)
            .filter(**lookup)
            [offset:offset + limit + 1]
        )
        more = len(data) > limit
        if more:
            data = data[:-1]
        count = len(data)

        return self.OK(
            'Listing badges',
            data={'badges': data, 'count': count, 'more': more}
        )


def _check_access_rights(user, owner, permdict):
    perm = permdict[owner]
    if not user.has_perm(perm):
        data = {'premission_required': perm}
        # TODO: logging
        raise apilib.ApiMethod.UNAUTHORIZED(data=data)


def check_edit_rights(user, owner):
    return _check_access_rights(user, owner, OWNER_EDIT_PERMS)


def check_view_rights(user, owner):
    return _check_access_rights(user, owner, OWNER_VIEW_PERMS)


class BadgeErrorMixin(object):
    def _make_badge_error_response(self, error, code, desc, badge):
        return self.CUSTOM_RESPONSE({
            'errors': [{
                'code': code,
                'desc': desc,
            }],
            'content': {
                'id': {
                    'code': error.__class__.__name__,
                    'desc': error.__doc__,
                    'state': badge.state,
                }
            }
        })


@RfidApi.register
class ActivateBadge(apilib.ApiMethod, BadgeErrorMixin):
    url = r'^badges/activate.{format}/?$'
    help = 'Activate a badge'

    badge_id = InputParam('Badge id', name='id', required=True,
                          validator=t.Int(gt=0))

    def POST(self):
        author = self.request.user.get_profile()
        try:
            badge = Badges(author=author).get(id=self.p.badge_id)
        except DoesNotExist:
            return self.NOT_FOUND()

        check_edit_rights(self.request.user, badge.owner)

        try:
            badge.activate()
        except RfidUnavailableStateTransition as error:
            return self._make_badge_error_response(
                error, 'activate_error', 'Unable to activate this badge.',
                badge
            )

        data = {
            'modified': badge.updated_at.isoformat().split('T')[0],
            'state': badge.state,
        }
        return self.OK('Badge activated.', data=data)


@RfidApi.register
class DeactivateBadge(apilib.ApiMethod, BadgeErrorMixin):
    url = r'^badges/deactivate.{format}/?$'
    help = 'Deactivate a badge'

    badge_id = InputParam('Badge id', name='id', required=True,
                          validator=t.Int(gt=0))

    def POST(self):
        author = self.request.user.get_profile()
        try:
            badge = Badges(author=author).get(id=self.p.badge_id)
        except DoesNotExist:
            return self.NOT_FOUND()

        check_edit_rights(self.request.user, badge.owner)

        try:
            badge.deactivate()
        except RfidUnavailableStateTransition as error:
            return self._make_badge_error_response(
                error, 'deactivate_error', 'Unable to deactivate this badge.',
                badge
            )

        data = {
            'modified': badge.updated_at.isoformat().split('T')[0],
            'state': badge.state,
        }
        return self.OK('Badge deactivated.', data=data)


@RfidApi.register
class BlockBadge(apilib.ApiMethod, BadgeErrorMixin):
    url = r'^badges/block.{format}/?$'
    help = 'Mark badge as blocked in LENEL'

    badge_id = InputParam('Badge id', validator=t.Int(gt=0), name='id',
                          required=True)

    def POST(self):
        author = self.request.user.get_profile()
        try:
            badge = Badges(author=author).get(id=self.p.badge_id)
        except DoesNotExist:
            return self.NOT_FOUND()

        try:
            badge.block()
        except RfidUnavailableStateTransition as error:
            return self._make_badge_error_response(
                error, 'block_error', 'Unable to block this badge.', badge
            )

        data = {
            'modified': badge.updated_at.isoformat().split('T')[0],
            'state': badge.state,
        }
        return self.OK('Badge blocked.', data=data)


@RfidApi.register
class DeleteBadge(apilib.ApiMethod, BadgeErrorMixin):
    url = r'^badges/delete.{format}/?$'
    help = 'Completely delete badge'

    badge_id = InputParam('Badge id', name='id', required=True,
                          validator=t.Int(gt=0))

    def POST(self):
        try:
            badge = Badges().get(id=self.p.badge_id)
        except DoesNotExist:
            return self.NOT_FOUND()

        check_edit_rights(self.request.user, badge.owner)

        try:
            badge.delete()
        except RfidUnavailableStateTransition as error:
            return self._make_badge_error_response(
                error, 'delete_error', 'Unable to delete this badge.',
                badge
            )
        return self.OK('Badge deleted.', data={})


@RfidApi.register
class BadgeEmployeeCreateEdit(apilib.ApiMethod, CodeErrorMixin):
    help = 'Create or edit employee badge.'
    url = r'^badges/employee.{format}/?$'
    require_permissions = ['rfid.edit_employees']

    badge_id = InputParam('Badge id', validator=t.Int(gt=0), name='id')
    staff = InputParam('Staff login', validator=StaffLoginValidator,
                       name='login', required=True)
    code = InputParam('Code to add', validator=CodeValidator)

    def POST(self):
        author = self.request.user.get_profile()
        if self.p.badge_id:
            # Edit
            try:
                badge = Badges(author=author).get(id=self.p.badge_id)
            except DoesNotExist:
                return self.NOT_FOUND()

            try:
                badge.code = self.p.code
            except RfidCodeError as error:
                return self._make_code_error_response(
                    error, 'employee_edit_error',
                    'Unable to edit this badge in such manner.'
                )

            badge.save()

        else:
            # New badge
            try:
                badge = Badges(author=author).create_employee(
                    person=self.p.staff,
                    code=self.p.code,
                )
            except RfidCodeError as error:
                return self._make_code_error_response(
                    error, 'employee_create_error',
                    'Unable to create badge with this code.'
                )

            except RfidPersonHasBadgeError as error:
                return self.CUSTOM_RESPONSE({
                    'errors': [{
                        'code': 'employee_create_error',
                        'desc': 'Unable to create new badge for this person.'
                    }],
                    'content': {
                        'login': [
                            {
                                'code': error.__class__.__name__,
                                'desc': error.__doc__,
                            }
                        ]
                    }
                })

        return self.OK(data={'id': badge.id})


@RfidApi.register
class BadgeShow(apilib.ApiMethod):
    help = 'View badge of any type.'
    url = r'^badges/show.{format}/?$'

    badge_id = InputParam('Badge id', validator=t.Int(gt=0), name='id',
                          required=True)

    def GET(self):
        try:
            badge = Badges().get(id=self.p.badge_id)
        except DoesNotExist:
            raise self.NOT_FOUND()

        check_view_rights(self.request.user, badge.owner)
        data = badge.as_dict(
            'id',
            'code',
            'state',
            'owner',
            'login',
            'reason',
            'created_at',
            'updated_at',
            'first_name',
            'first_name_en',
            'last_name',
            'last_name_en',
            'contractor_name',
            'contractor_id',
            'preprofile_id',
            'photo',
            'anonym_food_allowed',
            'changed_by',
        )
        brackets_regex = r'\(.*\)'
        for last_name in ('last_name', 'last_name_en'):
            if data[last_name]:
                data[last_name] = re.sub(brackets_regex, '', data[last_name])

        data['created_at'] = data['created_at'].isoformat().split('T')[0]
        data['updated_at'] = data['updated_at'].isoformat().split('T')[0]
        if data['changed_by']:
            data['changed_by'] = {
                'first_name': data['changed_by'].i_first_name,
                'last_name': data['changed_by'].i_last_name,
                'login': data['changed_by'].login,
            }
        return self.OK(data=data)


@RfidApi.register
class BadgeAnonymCreateEdit(apilib.ApiMethod, CodeErrorMixin):
    help = 'Create or edit anonym badge'
    url = r'^badges/anonymous.{format}/?$'
    require_permissions = ['rfid.edit_anonymous']

    badge_id = InputParam(
        'Badge id, leave blank to create new.', name='id',
        validator=t.Int(gt=0)
    )
    anonym_food_allowed = InputParam('Food allowed', validator=t.String,
                                     required=False)

    first_name = InputParam('First name', t.String, required=True)
    last_name = InputParam('Last name', t.String, required=True)

    org = InputParam('Organisation', OrgValidator, name='contractor')
    code = InputParam('Code', validator=CodeValidator)

    photo = InputParam(
        'Photo file', file_validator(), name='photo_file', required=False
    )

    def POST(self):
        author = self.request.user.get_profile()
        if not self.p.badge_id:
            # New badge
            try:
                badge = Badges(author=author).create_anonym(
                    first_name=self.p.first_name,
                    last_name=self.p.last_name,
                    contractor=self.p.org,
                    code=self.p.code,
                    photo_file=self.p.photo
                )
            except RfidCodeError as error:
                return self._make_code_error_response(
                    error, 'anonymous_create_error',
                    'Unable to create badge with this code.'
                )

        else:
            # Edit
            try:
                badge = Badges(author=author).get(id=self.p.badge_id)
            except DoesNotExist:
                return self.NOT_FOUND()

            if self.request.user.has_perm('rfid.change_anonym_food'):
                badge.anonym_food_allowed = self.p.anonym_food_allowed == 'true'

            badge.first_name = self.p.first_name
            badge.last_name = self.p.last_name
            badge.contractor = self.p.org
            try:
                badge.code = self.p.code
            except RfidCodeError as error:
                return self._make_code_error_response(
                    error, 'anonymous_edit_error',
                    'Unable to edit this badge in such manner.'
                )

            if self.p.photo and not isinstance(self.p.photo, str):
                badge.photo_file = self.p.photo

            try:
                badge.save()
            except AvatarBadRequestError as error:
                return self.CUSTOM_RESPONSE({
                    'errors': [{
                        'code': 'avatar_error',
                        'desc': 'Unable to upload image to Avatars.'
                    }],
                    'content': {
                        'photo_file': [
                            {
                                'code': error.__class__.__name__,
                                'desc': error.__doc__,
                                'status_code': error.code,
                            }
                        ]
                    }
                })

        return self.OK(data={'id': badge.obj.id})


@RfidApi.register
class ListCandidates(apilib.ApiMethod):
    url = r'^newcandidates/list.{format}'
    help = 'List candidates from newhire'
    require_permissions = ['rfid.list_candidates']

    limit = InputParam('Max results', validator=t.Int(gt=0, lte=100),
                       required=True)
    offset = InputParam('Results offset', validator=t.Int(gte=0))

    SORT = OrderedChoices(
        ('ALFA', 'alfa', 'Lexicographic order'),
        ('JOIN', 'join', 'Join date'),
    )

    search = InputParam('Search string', validator=t.String)
    sort = InputParam("Sort method (either 'alfa' or 'join')",
                      validator=ChoiceValidator(SORT))
    office_id = InputParam('From this office only',
                           validator=t.Int(gt=0) | magic_all)

    def GET(self):
        order = {
            'alfa': ('last_name',),
            'join': ('join_at', 'last_name'),
        }.get(self.p.sort or 'join')

        offset = self.p.offset or 0
        limit = self.p.limit

        lookup = {'office_id': self.p.office_id} if self.p.office_id else {}

        candidates = list(
            NewCandidates(self.request.user.get_profile())
            .actual()
            .filter(**lookup)
            .search(self.p.search)
            .order_by(*order)
            .values(
                'preprofile_id',
                'photo',
                'join_at',
                'office_id',
                'recruiter_id',
                'login',
                'first_name',
                'last_name',
            )
            [offset:offset + limit + 1]
        )

        has_more = len(candidates) > limit
        if has_more:
            candidates = candidates[:-1]

        self.set_weeks(candidates)
        self.join_related(candidates)

        count = len(candidates)

        data = {
            'count': count,
            'candidates': candidates,
            'more': has_more,
        }

        return self.OK(data=data)

    def set_weeks(self, candidates):
        today = date.today()
        current_ts = today + timedelta(7 - today.weekday())
        next_ts = current_ts + timedelta(7)

        for candidate in candidates:
            join_at = candidate['join_at']
            week = 'future'
            if join_at < current_ts:
                week = 'current'
            elif join_at < next_ts:
                week = 'next'

            candidate['week'] = week

    def join_related(self, candidates):
        offices_ids = set()
        recruiters_ids = set()
        preprofiles_ids = set()
        for candidate in candidates:
            offices_ids.add(candidate['office_id'])
            recruiters_ids.add(candidate['recruiter_id'])
            preprofiles_ids.add(candidate['preprofile_id'])

        recruiters = (
            Staff.objects
            .filter(id__in=recruiters_ids)
            .values(
                'id',
                'first_name',
                'last_name',
                'first_name_en',
                'last_name_en',
                'login',
            )
        )
        recruiters = {r['id']: localize(r) for r in recruiters}

        avatars = dict(
            AvatarMetadata.objects
            .filter(preprofile_id__in=preprofiles_ids)
            .values_list('preprofile_id', 'id')
        )

        offices = (
            Office.objects
            .filter(id__in=offices_ids)
            .values('id', 'name', 'name_en')
        )
        offices = {r['id']: localize(r) for r in offices}

        for candidate in candidates:
            recruiter_id = candidate.pop('recruiter_id', None)
            if recruiter_id in recruiters:
                candidate['recruiter'] = recruiters[recruiter_id]

            office_id = candidate.pop('office_id', None)
            if office_id in offices:
                candidate['office'] = offices[office_id]

            candidate['photo'] = None
            if candidate['preprofile_id'] in avatars:
                candidate['photo'] = avatars[candidate['preprofile_id']]


@RfidApi.register
class ShowNewCandidate(apilib.ApiMethod):
    url = 'newcandidates/show.{format}'
    help = 'Show candidate from Newhire'
    require_permissions = ['rfid.edit_candidates']

    preprofile_id = InputParam('Preprofile id', validator=t.Int(gt=0))

    def GET(self):
        fields = ('id', 'photo', 'join_at', 'office_id',
                  'first_name', 'first_name_en', 'last_name', 'last_name_en', 'login')
        try:
            candidate = (
                NewCandidates(self.request.user.get_profile())
                .actual()
                .values(*fields)
                .get(id=self.p.preprofile_id)
            )
        except Preprofile.DoesNotExist:
            return self.NOT_FOUND()
        candidate_data = candidate.obj
        candidate_data['preprofile_id'] = candidate_data.pop('id')
        return self.OK('Candidate info', data=candidate_data)


@RfidApi.register
class CandidateCreateEdit(apilib.ApiMethod, CodeErrorMixin):
    help = 'Edit candidate badge'
    url = r'^badges/candidate.{format}/?$'
    require_permissions = ['rfid.edit_candidates']

    id = InputParam('Badge id', validator=t.Int(gt=0))
    code = InputParam('Code', validator=CodeValidator)
    preprofile_id = InputParam('Preprofile id', validator=t.Int(gt=0))

    @staticmethod
    def format_candidate(badge):
        return dict(
            id=badge.id,
            code=badge.code,
            fullname=badge.full_name,
            login=badge.login,
            photo=badge.photo,
            office_id=badge.office_id,
            preprofile_id=badge.preprofile_id,
        )

    def POST(self):
        author = self.request.user.get_profile()
        if self.p.preprofile_id:
            try:
                candidate = (
                    NewCandidates(self.request.user.get_profile())
                    .actual()
                    .get(id=self.p.preprofile_id)
                    .to_candidate(code=self.p.code)
                )
                return self.OK(data=candidate.as_dict('id'))
            except DoesNotExist:
                return self.NOT_FOUND()
            except RfidCodeError as error:
                return self._make_code_error_response(
                    error, 'candidate_create_error',
                    'Unable to create badge with this code.'
                )
            except RfidCandidateBadgeAlreadyExist as error:
                return self._make_code_error_response(
                    error, 'candidate_create_error',
                    'This candidate has badge.'
                )
        else:
            try:
                candidate = Badges(author=author).get(id=self.p.id)
            except DoesNotExist:
                return self.NOT_FOUND()

            try:
                candidate.code = self.p.code
            except RfidCodeRewriteError as error:
                return self._make_code_error_response(
                    error, 'candidate_edit_error',
                    'Unable to edit this badge in such manner.'
                )

            candidate.save()

            return self.OK(data={
                candidate.id: self.format_candidate(candidate)
            })


@available_for_external('rfid.robot_with_rfid_api_access')
def sexport(request):
    if not request.user.has_perm('rfid.use_list_handle'):
        return HttpResponse(u'Access denied.\n', status=403)

    not_exported_anonymous_badges = (
        Q(owner=OWNER.ANONYM) & Q(anonym_food_allowed=False)
    )
    employee_without_guid = (
        Q(owner=OWNER.EMPLOYEE) & Q(person__guid=None)
    )

    badges = itertools.chain(
        Badges()
        .exclude(state=STATE.NOCODE)
        .exclude(not_exported_anonymous_badges)
        .exclude(employee_without_guid)
        .values(
            'login', 'guid', 'code', 'state', 'first_name', 'last_name',
            'owner', 'anonym_food_allowed',
        ),
        Reserves().filter(is_active=True).values('code')
    )

    def get_code(record):
        """ Определение типа записи
            0 - Бейдж сотрудника и у него есть guid
            1 - Бейдж кандидата или резерв
            2 - Бейдж анонима и стоит признак anonym_food_allowed
        """
        # Резерв.
        if 'state' not in record:
            return '1'

        return {OWNER.EMPLOYEE: '0', OWNER.CANDIDATE: '1', OWNER.ANONYM: '2'}[record['owner']]

    def serialize_to_csv():
        for badge in badges:
            try:
                line = [
                    '' if badge.get('owner') == OWNER.CANDIDATE else badge.get('login') or '',
                    badge.get('guid', '') or '',
                    '',
                    badge['code'],
                    '1' if badge.get('state',
                                     STATE.ACTIVE) == STATE.ACTIVE else '0',
                    badge.get('first_name', ''),
                    badge.get('last_name', ''),
                    badge.get('owner', ''),
                    get_code(badge),
                ]

                line = ' ; '.join(line)
                yield line

            except Exception:
                logger.exception(
                    'Bad data in badge with code "%s"', badge.get('code')
                )

    lines = '\n'.join(serialize_to_csv())

    return HttpResponse(lines, content_type='text/plain; charset=UTF-8')


@responding_json
@available_for_external('rfid.robot_with_rfid_api_access')
def sexport_key(request, key):
    if not request.user.has_perm('rfid.use_key_handle'):
        return {'message': 'access_denied'}, 403

    try:
        key_int = int(key)
    except ValueError:
        return {'message': 'rfid.wrong_code'}, 404

    try:
        badge = (
            Badges()
            .values(
                'login',
                'guid',
                'full_name',
                'owner',
                'state',
            )
            .get(rfid__code=key_int)
        )
    except DoesNotExist:
        return {'message': 'not_found'}, 404

    result = {
        'login': badge['login'],
        'guid': badge['guid'],
        'type': 'card',
        'guestfio': badge['full_name'],
        'holdertype': badge['owner'],
        'status': 1 if badge['state'] == STATE.ACTIVE else 2,
    }
    return result


@responding_json
@available_for_external('rfid.robot_with_rfid_api_access')
def sexport_login(request, login):
    if not request.user.has_perm('rfid.use_key_handle'):
        return {'message': 'access_denied'}, 403

    badge = (
        Badges()
        .values(
            'login',
            'guid',
            'full_name',
            'owner',
            'state',
            'code',
        )
        .filter(Q(person__login=login) | Q(login=login))
        .filter(state=STATE.ACTIVE)
        .first()
    )

    if not badge:
        return {'message': 'not_found'}, 404

    result = {
        'login': badge['login'],
        'guid': badge['guid'],
        'type': 'card',
        'guestfio': badge['full_name'],
        'holdertype': badge['owner'],
        'status': 1,
        'code': badge['code'],
    }
    return result


@auth_by_tvm_only(['badgepay'])
def export_payers(request):
    from_id = int(request.GET.get('from_id', 0))
    count = 2000

    anonym_with_food_allowed = Q(owner=OWNER.ANONYM) & Q(anonym_food_allowed=True)
    employees = Q(owner=OWNER.EMPLOYEE, person__is_dismissed=False)
    badges = list(
        Badges()
        .filter(state=STATE.ACTIVE)
        .filter(anonym_with_food_allowed | employees)
        .values('id', 'code', 'state', 'owner', 'login', 'first_name', 'last_name', 'middle_name')
        .order_by('id')
        .filter(id__gte=from_id)[:count]
    )

    payer_badges = [PayerBadge(**badge) for badge in badges]
    result = PayersReport(payer_badges).create_report()
    response = make_json_response(request, result)

    if len(badges) == count:
        response['Link'] = '<{url}?from_id={next_from_id}>; rel="next"'.format(
            url=request.build_absolute_uri(request.path),
            next_from_id=badges[-1]['id'] + 1
        )

    return response
