from django.db import models
from django.db.models.query_utils import Q

from staff.security.objects import AccessControlListService as ACLService


class AvailableObjectsManager(models.Manager):
    """
    Для использования менеджера в модели должны быть внешние ключи на Department
    и Office из django_intranet_stuff.models
    """

    def available_for(self, staff):
        """
        Метод менеджера, фильтрующий сущности, доступные пользователю
        """
        queryset = self.get_queryset().exclude(staff=staff)

        if staff.user.is_superuser:
            return queryset

        # TODO: разобраться с мемкэшем
        query = self.get_query(staff)

        if query is True:
            return queryset
        elif query is False:
            return queryset.none()
        return queryset.filter(query).distinct()

    def get_query(self, staff):
        """
        Метод для формирования Q объекта для фильтрации. Находит все методы,
        начинающиеся с handle и склевает через OR возвращенные ими условия.

        handle-метод может возвращать:
         * Q-объект для фильтрации
         * True - если пользователю доступны все объекты
         * False - если пользователю не доступны никакие объекты
        """
        handlers = (attr for attr in dir(self) if attr.startswith('handle'))
        q_condintions = Q()
        for handler in handlers:
            q_cond = getattr(self, handler)(staff)

            # если один из handle-методов вернул True - значит пользователь
            # всемогущне - не фильтруем сущности
            if q_cond is True:
                return True

            # из собранных условий отбрасываем False, т.к. они не нужны для фильтра
            if q_cond:
                q_condintions |= q_cond

        # если остались какие-то условия, сцепляем через OR
        # иначе сигнализируем, что у пользователя нет никаких прав на объекты
        return q_condintions or False

    def handle_dataprofiles(self, staff):
        """
        Метод для префильтрации сущностей, которые доступны пользователю на
        основе списков контроля доступа (ACL), связанных с группами, в которые
        входит пользователь.
        """
        dataprofiles = ACLService(staff).get_all_dataprofiles()
        if not dataprofiles:
            return False

        query = Q()
        for dp in dataprofiles:

            q = dp.to_Q(staff)

            # если среди пользовательских профилей попался пустой (всемогущий)
            # профиль, тогда возвращаем True, сигнализируя о том, что не нужно
            # фильтровать объекты
            if not q:
                return True

            query |= q

        return query
