import yenv

from django.template import Template, Context
from django.core import mail
from django.conf import settings


class SendException(Exception):
    def __init__(self, message, fail_person, not_performed):
        super(SendException, self).__init__(message)
        self.fail_person = fail_person
        self.not_performed = not_performed


class SpamMessages(object):

    def __init__(self, logins, author, text, is_html):
        self.persons = logins
        self.author = author
        self.text = text
        self.is_html = is_html
        self._last_person = None
        self._last_iter = []

    def __iter__(self):
        self._last_iter = iter(self.persons)
        for person in self._last_iter:
            self._last_person = person
            yield self._make_message(person)

    def __len__(self):
        return len(self.persons)

    def _make_message(self, person):
        subject, body = self._render_email(person)

        email_to = [person.work_email]
        if yenv.type != 'production':
            email_to = [settings.DEBUG_EMAIL]

        email = mail.EmailMessage(
            subject=subject,
            body=body,
            from_email=self.author,
            to=email_to,
            headers={'X-SendSpam': True},
        )

        if self.is_html:
            email.content_subtype = "html"

        return email

    def _render_email(self, person):
        context = {'person': person}

        text = self.text
        if yenv.type != 'production':
            text += '\n\nOriginal-To: {{ person.work_email }}'

        template = Template(text)

        subject, body = template.render(Context(context)).split('\n\n', 1)
        return subject, body

    def send(self):
        connection = mail.get_connection()
        try:
            connection.send_messages(self)
            self._last_person = None
            self._last_iter = []
        except Exception as e:
            raise SendException(e, self._last_person, self._last_iter)
