"""Переналивка базы данных с дампа продакшеновой базы"""

import logging
import os
import random

import sh
import yenv
from django.conf import settings
from django.core.management import call_command

from staff.departments.controllers.proposal import get_mongo_objects
from staff.departments.models import Department, DepartmentStaff, ProposalMetadata
from staff.groups.models import Group, GroupMembership
from staff.person.models import Staff
from staff.person_avatar.models import AvatarMetadata, PreprofileExtraFields
from staff.preprofile.models import Preprofile
from staff.preprofile.models.person_adopt_application import PersonAdoptApplication
from staff.preprofile.models.preprofile import PreprofileABCServices
from staff.proposal.models import CityAttrs
from staff.rfid.models import Badge
from staff.users.models import User


logger = logging.getLogger('db_renew')

TMP_DIR = '/tmp/staff_db'
DUMP_NAME = 'staff4.data.sql.gz'
BACKUP_HOST = 'vs-dbbackup01h.yandex.ru'
BACKUP_PATH = 'mysql_staff4_backup'

SUPERUSERS = [
    'aisel',
    'aksenovma',
    'amiroshnik',
    'baranovxyz',
    'cracker',
    'd1568',
    'denis-p',
    'dima117a',
    'dmirain',
    'grikyan91',
    'grisshafkn',
    'rchernitsyn',
    'remnev',
    'robot-staff-autotest',
    'shigarus',
    'tavria',
    'terrmit',
    'user3370',
    'user3993',
    'wlame',
]
ADMINS = [
    'user3370',
]
DEPARTMENT_CHIEFS = [
    ('user3370', 'virtual_robot'),
    ('user3370', 'yandex_ctoffice'),
    ('user3371', 'yandex_proffice'),
    ('robot-staff-autotest', 'yandex_content'),
]

FIXTURES_PATH = '/snippets/testing_fixtures/'


def _set_admins():
    print('\tset admins')
    for admin_login in ADMINS:
        GroupMembership.objects.get_or_create(
            staff=Staff.objects.get(login=admin_login),
            group=Group.objects.get(name='Admins')
        )


def _set_superusers():
    print('\tset superusers')
    (
        User.objects
        .filter(username__in=SUPERUSERS)
        .update(is_superuser=True, is_staff=True)
    )


def _set_department_chiefs():  # STAFF-4419
    print('\tset department chiefs')
    for login, dep_url in DEPARTMENT_CHIEFS:
        DepartmentStaff.objects.filter(
            department=Department.objects.get(url=dep_url),
            role_id='C',
        ).delete()
        DepartmentStaff.objects.get_or_create(
            staff=Staff.objects.get(login=login),
            department=Department.objects.get(url=dep_url),
            role_id='C',
        )


def _move_arnold_to_EKB():
    print('\tmove arnold to EKB')
    Staff.objects.filter(login='user3371').update(office=3)


def _move_avtotest_robot_to_MSK():
    print('\tmove robot-staff-autotest to EKB')
    Staff.objects.filter(login='robot-staff-autotest').update(office=1)


def _unrobot_zomb_prj_204():
    Staff.objects.filter(login='zomb-prj-204').update(is_robot=False)


def _unrobot_robot_staff_autotest():
    Staff.objects.filter(login='robot-staff-autotest').update(is_robot=False)


def _set_st_components():
    print('\tset startrek components')
    testing_ids = {
        'Берлин': 25665,
        'Екатеринбург': 25482,
        'Казань': 25666,
        'Киев': 25667,
        'Минск': 25668,
        'Москва': 25483,
        'Нижний Новгород': 25669,
        'Новосибирск': 25670,
        'Ростов-на-Дону': 25481,
        'Санкт-Петербург': 25671,
        'Симферополь': 25672,
        'Стамбул': 25673,
        'Пало-Альто': 25674,
    }

    for city_attr in CityAttrs.objects.filter(city__name__in=testing_ids.keys()):
        city_attr.component = testing_ids[city_attr.city.name]
        city_attr.save()

    CityAttrs.objects.filter(city=None).update(component=44620)


def _fake_proposal_metadata():
    """удаляем продакшеновые ProposalMetadata
    и придумываем рандомные для существующих заявок в монге
    все помечаем выполненными и запушенными в оебс.
    """
    print('\tfake proposal metadata')
    spec = dict.fromkeys(('author', 'created_at'), {'$exists': True})
    existing_in_mongo = []
    for mongo_obj in get_mongo_objects(spec, limit=0):
        pid = str(mongo_obj['_id'])
        try:
            pmd = ProposalMetadata.objects.get(proposal_id=pid)
        except ProposalMetadata.DoesNotExist:
            pmd = ProposalMetadata.objects.create(
                proposal_id=pid,
                applied_at=mongo_obj['created_at'],
                pushed_to_oebs=mongo_obj['created_at'],
                applied_by_id=int(mongo_obj['author']),
            )
        existing_in_mongo.append(pmd.id)
    ProposalMetadata.objects.exclude(id__in=existing_in_mongo).delete()


def _fake_preprofile_links():
    """
    Так как табилчку препрофайлов не наливаем, надо все
    таблички со ссылками на препрофайл проапдейтить
    существующими preprofile_id
    """
    print('\tfake preprofile links')
    existing_ids = list(Preprofile.objects.values_list('id', flat=True))

    print('\t\t - Badge')
    for badge in Badge.objects.exclude(preprofile_id=None):
        badge.preprofile_id = random.choice(existing_ids)
        badge.save()

    print('\t\t - PersonAdoptApplication')
    for paa in PersonAdoptApplication.objects.exclude(preprofile_id=None):
        paa.preprofile_id = random.choice(existing_ids)
        paa.save()

    print('\t\t - AvatarMetadata')
    for amd in AvatarMetadata.objects.exclude(preprofile_id=None):
        amd.preprofile_id = random.choice(existing_ids)
        amd.save()

    print('\t\t - PreprofileExtraFields')
    for pef in PreprofileExtraFields.objects.exclude(preprofile_id=None):
        pef.preprofile_id = random.choice(existing_ids)
        pef.save()

    print('\t\t - PreprofileABCServices')
    for pas in PreprofileABCServices.objects.exclude(preprofile_id=None):
        pas.preprofile_id = random.choice(existing_ids)
        pas.save()


def set_initial_data():
    print('Setting initial data')
    _set_admins()
    _set_superusers()
    _set_department_chiefs()
    _move_arnold_to_EKB()
    _move_avtotest_robot_to_MSK()
    _unrobot_zomb_prj_204()
    _unrobot_robot_staff_autotest()
    _set_st_components()
    _fake_proposal_metadata()
    _fake_preprofile_links()
    print('\nsetting initial data done.')


def load_fixtures():
    fixtures_dir = settings.PROJECT_PATH + FIXTURES_PATH
    for fixture_name in os.listdir(fixtures_dir):
        print('Load fixtures from {path}'.format(path=fixtures_dir + fixture_name))
        call_command('loaddata', fixtures_dir + fixture_name)


def unzip(filename):
    print('unzipping (It\'ll take 6-10 min) TODO: recalculate')
    sh.gzip('-d', filename)


def exclude_tables(dump_filename, new_filename):
    tables_to_exclude = [
        'preprofile_preprofile',
        'vcard_personvcard',
        'achievery_icon',
        'audit_log',
        'babylon_log',
        'notifications_messagethread',
        'oebs_storedresponse',
        'budget_position_changeregistry',
        'budget_position_workflow',
        'departments_headcountposition',
        'oebs_headcountposition',
    ]

    SKIPPING, SKIPPED = 0, 1
    state = SKIPPED
    with open(dump_filename, 'r') as infile, open(new_filename, 'a') as outfile:
        for line in infile:
            if line[:20] == 'DROP TABLE IF EXISTS':
                if line[22:-3] in tables_to_exclude:
                    state = SKIPPING
                    continue
                if state == SKIPPING:
                    state = SKIPPED

            if state == SKIPPING:
                continue

            outfile.write(line)


def rollup_db(dump_filename):
    assert yenv.type != 'production'
    print('rollupping (It\'ll take ~40 min)')
    result = sh.psql(
        sh.cat(dump_filename, _piped=True),
        '-u{username}'.format(username=settings.DATABASES['default']['USER']),
        '-p{password}'.format(password=settings.DATABASES['default']['PASSWORD']),
        '--host={host}'.format(host=settings.DATABASES['default']['HOST']),
        '--database={db}'.format(db=settings.DATABASES['default']['NAME']),
        '--port={port}'.format(port=settings.DATABASES['default']['PORT']),
    )
    print('STDERR:\n{}'.format(result.stderr))
    print('STDOUT:\n{}'.format(result.stdout))
    assert result.exit_code == 0, 'Rollup process failed'


def run(*args, **kwargs):
    zipped_dump_filename = 'staff4.data.sql.gz'

    unzip(zipped_dump_filename)

    dump_filename = zipped_dump_filename[:-3]
    new_filename = 'dump.sql'

    exclude_tables(dump_filename, new_filename)

    rollup_db(new_filename)

    set_initial_data()

    print('Done.')
