import logging

from django.utils.functional import cached_property

from staff.departments.models.department import Department
from staff.stats.libraries import register_dictionary
from staff.stats.enums import OrganizationTypes, OfficeTypes
from staff.stats.statface import statface
logger = logging.getLogger(__name__)


class Dictionary:
    name = None
    # динамические словари строятся по данным из бд и со временем могут меняться
    # поэтому их стоит отдельно пометить эти флагом и периодически подгружать в stat
    is_dynamic = False

    @cached_property
    def api(self):
        full_name = f'staff_{self.name}'
        return statface.get_stat_dict(full_name)

    def download(self) -> dict:
        """
        Загружает словарь из Statface
        :return:
        """
        return self.api.download()

    def upload(self):
        """
        Заливает словарь на Statface
        """
        return self.api.upload(self.get())

    def get(self):
        """
        Формирует словарь
        :return:
        """
        return {}


@register_dictionary
class DepartmentsDict(Dictionary):

    name = 'departments_dict'
    is_dynamic = True

    def get(self):
        return dict(Department.objects.values_list('id', 'name'))


@register_dictionary
class OrganizationType(Dictionary):

    name = 'organization_types_dict'

    def get(self):
        return {
            OrganizationTypes.yandex_organization: 'ЮЛ группы компаний Яндекс',
            OrganizationTypes.no_organization: 'Вне организаций',
            OrganizationTypes.yandex_money: 'Яндекс Деньги',
        }


@register_dictionary
class OfficeType(Dictionary):

    name = 'office_types_dict'

    def get(self):
        return {
            OfficeTypes.virtual: 'Виртуальные',
            OfficeTypes.homie: 'Надомники',
            OfficeTypes.office: 'Все офисы',
        }
