from django.db.models import Q
from django.conf import settings

from staff.stats.fetchers.base import HierarchicReportDataFetcher
from staff.stats.utils import DepartmentUnit
from staff.stats.enums import OfficeTypes, OrganizationTypes, UNKNOWN
from staff.preprofile.models.preprofile import Preprofile, PREPROFILE_STATUS
from staff.person.models.person import Staff


class FemidaPreprofileDataFetcher(HierarchicReportDataFetcher):

    def collect_data(self):
        queryset = (
            Preprofile.objects
            .filter(Q(created_at__gte=self.from_dt) | Q(join_at__gte=self.from_dt))
            .values(
                'status',
                'login',
                'organization_id',
                'office_id',
                'office__is_virtual',
                'department',
                'femida_offer_id',
                'created_at',
                'join_at',
            )
        )

        logins = {p['login'] for p in queryset if p['status'] == PREPROFILE_STATUS.CLOSED}
        login_to_join_date = dict(
            Staff.objects
            .filter(login__in=logins)
            .values_list('login', 'join_at')
        )

        for preprofile in queryset:
            department = DepartmentUnit(preprofile['department'])
            office = self.get_office_type(preprofile)
            organization = self.get_organization_type(preprofile)

            keys = self.get_related_keys((
                department,
                organization,
                office,
            ))

            preprofile['join_at'] = login_to_join_date.get(preprofile['login'])

            for key in keys:
                self.result[key]['preprofiles_created'] += (
                    preprofile['created_at'] >= self.from_dt
                )
                self.result[key]['preprofiles_closed'] += (
                    preprofile['join_at'] is not None
                    and self.from_dt.date() <= preprofile['join_at'] < self.to_dt.date()
                )
                self.result[key]['femida_preprofiles_created'] += (
                    preprofile['created_at'] >= self.from_dt
                    and preprofile['femida_offer_id'] is not None
                )
                self.result[key]['femida_preprofiles_closed'] += (
                    preprofile['join_at'] is not None
                    and self.from_dt.date() <= preprofile['join_at'] < self.to_dt.date()
                    and preprofile['femida_offer_id'] is not None
                )

    @staticmethod
    def get_office_type(preprofile):
        if preprofile['office__is_virtual']:
            office = OfficeTypes.virtual
        elif preprofile['office_id'] == settings.HOMIE_OFFICE_ID:
            office = OfficeTypes.homie
        elif preprofile['office_id'] is not None:
            office = OfficeTypes.office
        else:
            office = UNKNOWN
        return office

    @staticmethod
    def get_organization_type(preprofile):
        if preprofile['organization_id'] == settings.ROBOTS_ORGANIZATION_ID:
            organization = OrganizationTypes.no_organization
        elif preprofile['organization_id'] == settings.YAMONEY_ORGANIZATION_ID:
            organization = OrganizationTypes.yandex_money
        elif preprofile['organization_id'] is not None:
            organization = OrganizationTypes.yandex_organization
        else:
            organization = UNKNOWN
        return organization
