import logging

from staff.celery_app import app
from staff.lib.tasks import LockedTask
from staff.stats.libraries import dictionaries_library, reports_library

logger = logging.getLogger(__name__)


def get_retry_countdown(retry):
    retry_periods = [5, 55, 240, 1500, 9000, 10800]
    return retry_periods[min(len(retry_periods) - 1, retry)]


@app.task(max_retries=5)
class UploadDictionariesToStat(LockedTask):
    def locked_run(self, dict_names=None, only_dynamic=True):
        failed_dict_names = []
        if dict_names is None:
            if only_dynamic:
                dict_names = dictionaries_library.dynamic_dictionaries.keys()
            else:
                dict_names = dictionaries_library.keys()

        logger.info('Uploading %d dictionaries to Stat: %s', len(dict_names), ', '.join(dict_names))

        for dict_name in dict_names:
            dict_class = dictionaries_library[dict_name]
            try:
                dict_class().upload()
            except Exception:
                logger.exception('Failed during upload `%s` dictionary to Stat', dict_name)
                failed_dict_names.append(dict_name)

        if failed_dict_names:
            self.retry(
                countdown=get_retry_countdown(self.request.retries),
                kwargs={
                    'dict_names': failed_dict_names,
                    'only_dynamic': only_dynamic,
                }
            )


@app.task(max_retries=5)
class UploadReportsToStat(LockedTask):
    def locked_run(self, report_names=None):
        failed_report_names = []

        if report_names is None:
            report_names = reports_library.keys()

        logger.info('Uploading %d reports to Stat: %s', len(report_names), ', '.join(report_names))

        for report_name in report_names:
            report_class = reports_library[report_name]
            try:
                report_class().upload_data()
            except Exception:
                logger.exception('Failed during upload `%s` report to Stat', report_name)
                failed_report_names.append(report_name)

        if failed_report_names:
            self.retry(
                countdown=get_retry_countdown(self.request.retries),
                kwargs={
                    'report_names': failed_report_names,
                }
            )
