import logging
import openpyxl
import xlrd
from datetime import date
from typing import Any, Dict, Optional, List

from staff.lib.db import atomic
from staff.map.models import COUNTRY_CODES
from staff.person.models import Staff


logger = logging.getLogger(__name__)


def parse_file(import_file: Any) -> Dict[str, int]:
    try:
        workbook = openpyxl.load_workbook(import_file)
        sheet = workbook.active
        rows = list(sheet.rows)
    except OSError:
        logger.info('Not an xlsx file %s, trying xls', import_file, exc_info=True)
        import_file.seek(0)
        workbook = xlrd.open_workbook(file_contents=import_file.read())
        sheet = workbook.sheet_by_index(0)
        rows = list(sheet.get_rows())

    if len(rows) < 1 or any(len(row) < 2 for row in rows):
        logger.info('Belarus vacation file has to few rows or columns')
        raise ValueError()

    login_index = 1
    leave_balance_index = 2
    header = rows[0]
    current_index = 0

    for cell in header:
        if str(cell.value).lower() in {'логин', 'login'}:
            login_index = current_index
        elif str(cell.value).lower() in {'остаток на дату отчета'}:
            leave_balance_index = current_index
        current_index += 1

    return {
        str(row[login_index].value).strip(): int(row[leave_balance_index].value)
        for row in rows[1:]
        if row[login_index].value is not None and row[leave_balance_index].value is not None
    }


def not_belarus(country: Optional[str]) -> bool:
    return str(country).lower() != COUNTRY_CODES.BELARUS.lower()


def upload_leave_balance(data: Dict[str, int]) -> Dict[str, List[str]]:
    today = date.today()
    countries = dict(Staff.objects.filter(login__in=data).values_list('login', 'organization__country_code'))
    missing = [login for login in set(data) - set(countries)]
    invalid_country = [login for login, country in countries.items() if not_belarus(country)]
    success = list(set(countries) - set(invalid_country))

    with atomic():
        for person in Staff.objects.filter(login__in=success).select_related('extra'):
            person.vacation = data[person.login]
            if hasattr(person, 'extra'):
                person.extra.last_vacation_accrual_at = today
                person.extra.save()
            person.save()
            logger.info('Updated vacation for %s to %s days', person.login, person.vacation)

    return {'missing': missing, 'invalid_country': invalid_country, 'success': success}
