from datetime import date
from typing import Any, Dict, List, Optional, Iterator

import attr
from dateutil import parser

from django.conf import settings

from staff.lib.requests import Session, HTTPError


st_session = Session()
st_session.headers.update({
    'Authorization': 'OAuth {token}'.format(token=settings.ROBOT_STAFF_OAUTH_TOKEN),
    'Content-type': 'application/json',
})


@attr.s(auto_attribs=True)
class Umbrella:
    issue_key: str
    name: str


@attr.s(auto_attribs=True)
class VS:
    issue_key: str
    abc_service_id: Optional[int]
    name: str
    umbrellas: List[Umbrella]
    deadline: date


class UmbrellasFetcher:
    def __init__(self) -> None:
        self.url = 'https://st-api.yandex-team.ru/v2/issues'  # always production
        self.per_page = 50
        self.timeout = (5, 10, 20)
        self.query = {
            'query': 'Queue:GOALZ Type:VS Status:!cancelled',
            'fields': ['summary', 'self', 'abcService', 'key', 'deadline'],
            'expand': 'links',
            'perPage': self.per_page,
        }

    def get_umbrellas_from_st(self) -> Iterator[VS]:
        raw_st_result = self._get_raw_data_from_st()
        return self._parse(raw_st_result)

    def _parse(self, raw_st_results: Iterator[Dict[str, Any]]) -> Iterator[VS]:
        for item in raw_st_results:
            if not item['deadline']:
                continue

            umbrellas = [
                Umbrella(
                    issue_key=link['object']['key'],
                    name=link['object']['display'],
                )
                for link in item.get('links', [])
                if link['type']['id'] == 'subtask'
            ]
            vs = VS(
                issue_key=item['key'],
                abc_service_id=next((int(service['id']) for service in item['abcService']), None),
                name=item['summary'],
                umbrellas=umbrellas,
                deadline=parser.parse(item['deadline']).date(),
            )
            yield vs

    def _get_page(self, page: int) -> List[Dict[str, Any]]:
        params: Dict[str, Any] = {'page': page, **self.query}
        response = st_session.get(self.url, params=params, timeout=self.timeout)
        if response.status_code == 400:
            errors = response.json()['errorMessages']
            error_text = errors and errors[0]
            raise HTTPError(error_text, response=response)

        response.raise_for_status()
        return response.json()

    def _get_raw_data_from_st(self) -> Iterator[Dict[str, Any]]:
        page_number = 1

        items = self._get_page(page_number)
        yield from items
        while len(items) == self.per_page:
            page_number += 1
            items = self._get_page(page_number)
            yield from items
