import pytz
from datetime import datetime

from django.conf import settings

from staff.lib.utils.date import parse_datetime

from staff.trip_questionary.utils import get_default_city_component
from staff.trip_questionary.models import CityAsStartrekComponent


local_tz = pytz.timezone(settings.TIME_ZONE)


def set_issue(obj, issue, event_type):
    exclude_fields = [
        'version',
        'parent',
        'components',
        'access',
        'resolvedBy',
        'summary',
        'description',
        'type',
        'updatedBy',
        'priority',
        'tags',
        'followers',
        'createdBy',
        'unique',
        'assignee',
        'queue',
        'updatedAt',
        'sla',
        'pendingReplyFrom',
    ]

    issue_dict = issue.as_dict()
    issue_ = {}

    st_datetime_keys = {
        # startrekKey : tripDataKey
        'travelStartDate': 'startDate',
        'travelEndDate': 'endDate',
    }
    st_date_keys = {
        'yaStart': 'taxiStartDate',
        'yaEnd': 'taxiEndDate',
    }

    for key in issue_dict.keys():
        if key in exclude_fields:
            continue
        value = issue_dict[key]
        if key == 'createdAt':
            value = parse_datetime(value).astimezone(local_tz).replace(tzinfo=None)

        if key in st_datetime_keys:
            key = st_datetime_keys.get(key)
            value = _parse_st_datetime(value)

        if key in st_date_keys:
            key = st_date_keys.get(key)
            value = _parse_st_datetime(value)
            value = value and value.date()

        issue_[key] = value

    obj[event_type + '_issue'] = issue_


def _parse_st_datetime(value):
    if not value:
        return None
    return local_tz.localize(parse_datetime(value)).replace(tzinfo=None)


def set_component(obj, issue, event_type):
    if event_type != 'trip':
        return

    components = (
        CityAsStartrekComponent.objects
        .filter(component__in=(c.as_dict()['id'] for c in issue['components']))
    )

    try:
        obj['st_component'] = components[0]
    except IndexError:
        obj['st_component'] = get_default_city_component()


class Startrek(object):

    @staticmethod
    def make_comment(trip_uuid, author, notification_cls, issue_key, is_personal, context):
        params = {
            'author': author,
            'issue_key': issue_key,
        }
        notification = notification_cls(
            target='STARTREK_COMMENT',
            context=context,
            trip_uuid=trip_uuid,
            is_personal=is_personal,
        )

        notification.send(**params)


def _creation_notification_sent_state(trip, state):
    trip.data['is_creation_notification_sent'] = state


def set_creation_notification_sent(trip):
    _creation_notification_sent_state(trip, True)


def unset_creation_notification_sent(trip):
    _creation_notification_sent_state(trip, False)


def set_employee_creation_access_notification_sent(trip, recipient, people):
    notificated_chiefs = trip.data.setdefault('notificated_chiefs', {})
    employees = notificated_chiefs.setdefault(str(recipient.pk), [])
    employees.extend(set(people) - set(employees))


def set_synced_time(trip):
    trip.data['last_sync'] = datetime.now()
