from uuid import uuid4

import requests

from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from staff.trip_questionary.models import TripPurpose

from staff.lib.requests import get_ids_repository

from staff.trip_questionary.control import Startrek
from staff.trip_questionary.controller.notifications import (
    StartrekAddCityCommentNotification,
    StartrekCommonCommentNotification,
    StartrekChangePurposeCommentNotification,
)
from staff.trip_questionary.forms import ARRIVE_DATE_TYPE
from staff.trip_questionary.controller.operations.base import OperationBase, lib
from staff.trip_questionary.controller.operations.issues.creation_base import get_st_purpose_fields


class StartrekCommentCreationOperation(OperationBase):

    is_personal = False
    event_type = 'trip'

    def run(self):
        params = self.get_params().copy()
        params['author'] = self.trip.data['author']
        params['employee_list'] = self.trip.data['employee_list']

        issue_keys = self.get_issue_key()

        if isinstance(issue_keys, str):
            issue_keys = [issue_keys]

        for key in issue_keys:
            self.make_comment(issue_key=key, params=params)

        super(StartrekCommentCreationOperation, self).run()

    def make_comment(self, issue_key, params):
        Startrek.make_comment(
            self.trip.uuid,
            self.trip.data['author'],
            self.notification_cls,
            {'filter': 'key:' + issue_key},
            self.is_personal,
            context=params
        )

    def get_issue_key(self):
        return self.trip.data[self.event_type + '_issue']['key']


@lib.register
class ReturnCityAddedOperation(StartrekCommentCreationOperation):
    def match_diff_preconditions(self):
        return (
            not self.TC.is_new
            and
            not self.TC.is_conf
            and
            self.TC.is_trip_toplevel_issue_created
            and
            self.TC.is_return_city_added
        )

    notification_cls = StartrekAddCityCommentNotification

    def get_params(self):
        return self.trip.data['city_list'][-1]


@lib.register
class CityDatesEditOperation(StartrekCommentCreationOperation):
    def match_diff_preconditions(self):
        return (
            not self.TC.is_conf
            and
            not self.TC.is_new_city
            and
            self.TC.is_trip_toplevel_issue_created
            and
            self.TC.is_city_dates_changed
        )

    notification_cls = StartrekCommonCommentNotification

    def get_params(self):
        departure_date = self.city_data['departure_date']
        arrive_date_type = self.city_data['city_arrive_date_type']
        city = self.city_data['city']
        return {
            'object': city,
            'field_name': _('trip.label.city-departure_date'),
            'value': '{date}({arr_type})'.format(
                date=departure_date,
                arr_type=ARRIVE_DATE_TYPE[arrive_date_type],
            )
        }


class EmployeeEditOperation(StartrekCommentCreationOperation):
    is_personal = True
    notification_cls = StartrekCommonCommentNotification

    def get_params(self):
        value = self.employee_data[self.field['key']]
        employee = self.employee_data['employee']
        component_city = self.operations_cache.city_as_component(employee)
        return {
            'object': employee,
            'field_name': self.field['name'],
            'value': value,
            'employee_data': self.employee_data,
            'component_city': component_city,
        }

    def get_issue_key(self):
        return self.employee_data[self.event_type + '_issue']['key']


@lib.register
class EmployeeDepartureDateEditOperation(EmployeeEditOperation):
    def match_diff_preconditions(self):
        return (
            not self.TC.is_conf
            and
            not self.TC.is_new_employee
            and
            self.TC.is_trip_employee_issue_created
            and
            self.TC.is_employee_departure_date_changed
        )

    field = {
        'key': 'departure_date',
        'name': _('trip.st_label.departure_date'),
    }


@lib.register
class EmployeeReturnDateEditOperation(EmployeeEditOperation):
    def match_diff_preconditions(self):
        return (
            not self.TC.is_conf
            and
            not self.TC.is_new_employee
            and
            self.TC.is_trip_employee_issue_created
            and
            self.TC.is_employee_return_date_changed
        )

    field = {
        'key': 'return_date',
        'name': _('trip.st_label.return_date'),
    }


@lib.register
class PurposeChangingOperation(StartrekCommentCreationOperation):

    notification_cls = StartrekChangePurposeCommentNotification

    def run(self):
        self.update_tickets_fields()
        super(PurposeChangingOperation, self).run()

    def update_tickets_fields(self):
        issues = get_ids_repository(
            service='startrek2',
            resource_type='issues',
            oauth_token=settings.ROBOT_STAFF_OAUTH_TOKEN,
            user_agent=settings.STAFF_USER_AGENT,
        )
        for ticket_key in self.get_issue_key():
            ticket = issues.get_one(
                {'filter': 'key:{ticket_key}'.format(ticket_key=ticket_key)}
            )
            ticket.update(**get_st_purpose_fields(self.purpose_ids))

    def get_params(self):
        return {
            'purposes': TripPurpose.objects.filter(id__in=self.purpose_ids)
        }

    @property
    def purpose_ids(self):
        return self.trip.data['purpose']

    def get_issue_key(self):
        all_issue_keys = getattr(self, 'all_issue_keys', None)
        if all_issue_keys is None:
            self.all_issue_keys = set()

            def trip_and_conf_issue_keys(entity):
                for ticket_type in ('trip_issue', 'conf_issue'):
                    if ticket_type in entity:
                        yield entity[ticket_type]['key']

            for ticket_key in trip_and_conf_issue_keys(self.trip.data):
                self.all_issue_keys.add(ticket_key)

            for employee_data in self.trip.data['employee_list']:
                for ticket_key in trip_and_conf_issue_keys(employee_data):
                    self.all_issue_keys.add(ticket_key)

        return self.all_issue_keys

    def match_diff_preconditions(self):
        return (
            self.TC.is_trip_purpose_changed
            and self.TC.is_all_personal_issues_created
            and self.TC.is_all_toplevel_issues_created
        )


class StartApproval(StartrekCommentCreationOperation):
    is_personal = True
    approval_author_login = settings.ROBOT_STAFF_LOGIN

    @property
    def employee(self):
        return self.employee_data['employee']

    def match_preconditions(self):
        return (
            not self.TC.is_employee_approval_started
        )

    def get_params(self):
        request = requests.Request(
            url='https://{}/tracker'.format(settings.OK_HOST),
            params={
                '_embedded': 1,
                'object_id': self.get_issue_key(),
                'uid': uuid4().hex,
                'users': self.approvers,
                'groups': self.approval_groups,
                'author': self.approval_author_login,
            }
        )

        return {
            'url': request.prepare().url,
            'employee_data': self.employee_data,
        }

    def get_issue_key(self):
        return self.employee_data[self.event_type + '_issue']['key']

    def set_approval_started(self):
        self.employee_data['is_approval_started'] = True

    def run(self):
        super(StartApproval, self).run()
        self.set_approval_started()
