import yenv
import logging

from django.conf import settings
from django.core.exceptions import ObjectDoesNotExist

from staff.lib.log import log_context
from staff.oebs.utils import get_oebs_assignments_by_person_logins
from staff.person.models import Staff

from staff.trip_questionary.models import DepartmentAsStartrekComponent
from staff.trip_questionary.models import CityAsStartrekComponent
from staff.trip_questionary.startrek import IssueGetter

from staff.trip_questionary.controller.operations.base import lib

logger = logging.getLogger('staff.trip_questionary.controller')


class OperationsCache(object):

    def __init__(self, trip):
        self.trip = trip
        self._city_as_component = None
        self.st_issues = IssueGetter(self.trip.data['author'])
        self.employee_assignments = None

    def get_employee_assignment(self, login):
        """ Назначения, выбранные в заявке.
        {
            login: {
                'assignmentID': 73220,
                'legislationCode': 'RU',
                'primaryFlag': 'Y',
                'taxUnitID': 121,
                'taxUnitName': 'ООО Яндекс',
                'orgID': 77652,
                'orgName': 'Группа разработки Стаффа и Фемиды',
                'contractType': 'Основное место работы',
            },
        }
        """
        if self.employee_assignments:
            return self.employee_assignments.get(login)

        self.employee_assignments = {}
        # по логину хранится id выбранного назначения в заявке
        employee_assignment_by_login = {  # здесь {login: assignment_id}
            e['employee'].login: e['employee_assignment']
            for e in self.trip.data['employee_list']
        }
        assignments = get_oebs_assignments_by_person_logins(list(employee_assignment_by_login.keys()))

        if not assignments:
            return None

        for login in employee_assignment_by_login:
            self.employee_assignments[login] = None

            if not assignments[login]:
                continue

            # всегда берем только первое назначение, потому что оно основное
            self.employee_assignments[login] = assignments[login][0]

        # если назначения не пришли то возвращает None
        # будет красный текст в тикете и невыполнен запрос в таксишный кабинет
        return self.employee_assignments.get(login)

    def city_as_component(self, person):
        if self._city_as_component is None:
            emp_pk = [
                e['employee'].pk
                for e in self.trip.data['employee_list']
            ]
            emp_list = (
                Staff.objects
                .select_related('office__city__cityasstartrekcomponent')
                .filter(pk__in=emp_pk)
            )

            component_dict = {None: CityAsStartrekComponent.objects.get(
                city_id=settings.DEFAULT_CITY_ID
            )}

            for emp in emp_list:
                try:
                    component = emp.office.city.cityasstartrekcomponent
                except (AttributeError, ObjectDoesNotExist):
                    component = component_dict[None]
                component_dict[emp] = component

            self._city_as_component = component_dict

        component = (
            self._city_as_component.get(person) or
            self._city_as_component.get(None)
        )
        return component

    @staticmethod
    def direction(person):
        direction_departments = list(
            person.department
            .get_ancestors(include_self=True)
            .filter(kind_id=settings.DIS_DIRECTION_KIND_ID)
            .order_by('level')
        )
        if len(direction_departments) == 0:
            return None
        return direction_departments[0]

    def direction_as_component(self, person):
        direction = self.direction(person)

        try:
            component = (
                DepartmentAsStartrekComponent.objects
                .get(department=direction)
                .component
            )

        except DepartmentAsStartrekComponent.DoesNotExist:
            component = None

        return component


class OperationsRegistry(object):

    def __init__(self, trip):
        self.trip = trip
        self.operations_cache = OperationsCache(trip)
        self.employee_list = self.trip.data['employee_list']
        self.city_list = self.trip.data.get('city_list', [])

    def _get_params(self):
        return {
            'trip': self.trip,
            'diff': self.trip.diff,
            'operations_cache': self.operations_cache,
        }

    def _get_emp_params(self, employee_index):
        params = self._get_params()
        if params['diff']['employee_list'].is_simple:
            diff = None
        else:
            diff = self.trip.diff['employee_list'][employee_index]

        employee_params = {
            'employee_data': self.employee_list[employee_index],
            'employee_diff': diff,
            'operations_cache': self.operations_cache,
        }

        employee_params.update(params)
        return employee_params

    def __iter__(self):

        yield lib.SyncIssues(**self._get_params())

        # Editing ###
        yield lib.ReturnCityAddedOperation(**self._get_params())

        yield lib.PurposeChangingOperation(**self._get_params())

        for city_index, city_data in enumerate(self.city_list):
            yield lib.CityDatesEditOperation(
                city_data=city_data,
                city_diff=self.trip.diff['city_list'][city_index],
                **self._get_params()
            )

        for i in range(len(self.employee_list)):
            yield lib.EmployeeDepartureDateEditOperation(**self._get_emp_params(i))
            yield lib.EmployeeReturnDateEditOperation(**self._get_emp_params(i))

        # Creation ###

        # Single:
        yield lib.SingleConfInterconf(**self._get_params())

        yield lib.SingleTripConfInterconf(**self._get_params())
        yield lib.SingleTripConfTravel(**self._get_params())

        yield lib.SingleTripTravel(**self._get_params())

        # Meta:
        yield lib.MetaConfInterconf(**self._get_params())

        yield lib.MetaTripConfInterconf(**self._get_params())
        yield lib.MetaTripConfTravel(**self._get_params())

        yield lib.MetaTripTravel(**self._get_params())

        # Personal
        for i in range(len(self.employee_list)):

            # Personal Startrek issues
            yield lib.PersonalConfInterconf(**self._get_emp_params(i))
            yield lib.PersonalTripConfInterconf(**self._get_emp_params(i))
            yield lib.PersonalTripConfTravel(**self._get_emp_params(i))
            yield lib.PersonalTripTravel(**self._get_emp_params(i))

            # Related Issues
            yield lib.HRMobile(**self._get_emp_params(i))

            # Personal emails
            yield lib.NoticeEmail(**self._get_emp_params(i))
            yield lib.OnTripIssueCloseEmail(**self._get_emp_params(i))

            # Taxi account
            yield lib.CreateTaxiAccount(**self._get_emp_params(i))
            yield lib.RemoveTaxiAccount(**self._get_emp_params(i))

            # Gaps in GAP
            yield lib.RenewEditGap(**self._get_emp_params(i))
            yield lib.UpdateEditGap(**self._get_emp_params(i))
            yield lib.CreateGap(**self._get_emp_params(i))
            yield lib.ConfirmEditGap(**self._get_emp_params(i))
            yield lib.CancelEditGap(**self._get_emp_params(i))

        # Link issues if TRIP+CONF
        yield lib.LinkTripConfIssues(**self._get_params())


def run_operations(trip):
    last_exception = None

    for operation in OperationsRegistry(trip):
        with log_context(operation_name=operation.name):
            try:
                if operation.match_preconditions():
                    logger.info(f'Running operation {type(operation)}')
                    operation.run()
                    trip.save()
                else:
                    logger.info(f'Preconditions not match {type(operation)}')

            except Exception as e:
                logger.exception('')
                last_exception = e
                if yenv.type == 'development':
                    raise

    trip.save()

    if last_exception is not None:
        raise last_exception

    logger.info('Trip operations completed')


def delay_diff_operations(trip):
    logger.info('Delay diff operations started.')

    for operation in OperationsRegistry(trip):
        with log_context(operation_name=operation.name):
            try:
                if operation.match_diff_preconditions():
                    logger.info('Delay operation')
                    operation.set_to_delayed()
                else:
                    logger.debug('Diff preconditions not match')

            except Exception:
                logger.exception('')
                if yenv.type == 'development':
                    raise

    logger.info('Delay diff operations complete.')
