import re
from typing import List

from functools import partial
from itertools import chain
from staff.django_phonenumbers import PhoneField

from django import forms
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext_lazy as _

from staff.person.models import Staff
from staff.lib.utils.ordered_choices import StaffChoices, Option

from staff.lib.utils.ordered_choices import OrderedChoices
from staff.trip_questionary.tree_form import (
    FIELD_STATE,
    TreeForm, TreeFormField, ModelChoiceField, MulticModelChoiceField,
    CharField, EmailField, BooleanField, NullBooleanField, ChoiceField,
    ValidChoiceField, IntegerField, ModelMultipleChoiceField)

from staff.trip_questionary.tree_form import DateField as DateField_
from .sorted import STAFF_TRIPS_QUERIES
from .controller.context import (
    get_issue_url,
    update_dates,
    get_departure_city,
    get_return_city)

from .models import TripPurpose, EVENT_TYPE

extended_event_type_params = [
    (o.name, o.value, o.desc) for o in chain(
        [Option('ALL', 'all', 'all types')],
        EVENT_TYPE._options,
    )
]
EVENT_TYPE_EXTENDED = OrderedChoices(*extended_event_type_params)

roles_choices = [(r.upper(), r.lower(), r.lower()) for r in STAFF_TRIPS_QUERIES]
EVENT_ROLES = OrderedChoices(*roles_choices)

DATE_INPUT_FORMATS = (
    '%d.%m.%Y',  # '25.10.2006'
    '%d.%m.%y',  # '25.10.06'
    '%Y-%m-%d',  # '2006-10-25'
    '%m/%d/%Y',  # '10/25/2006'
    '%m/%d/%y',  # '10/25/06'
)


DateField = partial(DateField_, input_formats=DATE_INPUT_FORMATS)


class CityModelChoiceField(ModelChoiceField):

    def choice_name_from_instance(self, obj):
        return obj.i_name


required_msg = _('This field is required.')


EMPTY_CHOICES = OrderedChoices()


TRANSPORT = OrderedChoices(
    ('AIRCRAFT', 'aircraft', _('trip.transport-aircraft')),
    ('TRAIN', 'train', _('trip.transport-train')),
    ('BUS', 'bus', _('trip.transport-bus')),
    ('CAR', 'car', _('trip.transport-car')),
)


FARE = OrderedChoices(
    ('MOST_ECONOMICAL', 'most_economical', _('trip.fare-most_economical')),
    ('WITHOUT_RETURN', 'without_return', _('trip.fare-without_return')),
    ('WITH_RETURN', 'with_return', _('trip.fare-with_return')),
)


COMPENSATION = OrderedChoices(
    ('MONEY', 'money', _('trip.compensation-money')),
    ('DAY_OFF', 'day_off', _('trip.compensation-day_off')),
)


BAGGAGE = OrderedChoices(
    ('HAND', 'hand', _('trip.baggage-hand')),
    ('HAND_PLUS', 'hand_plus', _('trip.baggage-hand_plus')),
)


EVENT_ROLE = StaffChoices(
    LISTENER=('listener', _('trip.event_role-listener')),
    SPEAKER=('speaker', _('trip.event_role-speaker')),
)

ARRIVE_DATE_TYPE = OrderedChoices(
    ('DEPARTURE', 'departure', _('trip.arrive_date_type.departure')),
    ('ARRIVAL', 'arrival', _('trip.arrive_date_type.arrival')),
)


STATUS_PRESENTATION = {
    ('open', None, False): _('trip.issue_status-new'),
    ('approved', None, False): _('trip.issue_status-approved'),
    ('closed', 'fixed', False): _('trip.issue_status-fixed'),
    ('closed', "won'tFix", False): _('trip.issue_status-wont'),
    ('checkIn', None, False): _('trip.issue_status-checkin'),
    ('open', None, True): _('trip.employee.issue_status-new'),
    ('approved', None, True): _('trip.employee.issue_status-approved'),
    ('closed', 'fixed', True): _('trip.employee.issue_status-fixed'),
    ('closed', "won'tFix", True): _('trip.employee.issue_status-wont'),
    ('checkIn', None, True): _('trip.issue_status-checkin'),
}


def get_status_presentation(data, event_type, is_employee):
    """Подробнее читать тут
    http://wiki.yandex-team.ru/staff/visions/travel/status-matches"""

    issue = data.get('{0}_issue'.format(event_type), {})

    status_key = issue.get('status', {}).get('key', None)
    resolution_key = issue.get('resolution') or {}
    resolution_key = resolution_key.get('key', None)

    presentation = STATUS_PRESENTATION.get(
        (status_key, resolution_key, is_employee),
        status_key or '',
    )
    return presentation


# Общие куски
class BaseEmployeeForm(TreeForm):
    """Базовая форма для элемента списка сотрудников."""

    employee = MulticModelChoiceField(
        Staff.objects.all(),
        label=_('trip.label.employee-employee'),
        help_text=_('trip.help.employee-employee'),
        state=FIELD_STATE.REQUIRED)

    def get_field_state(self, name):

        is_new = self.parent_form.initial.get('is_new')

        # Если не новая и уже есть сохраненная часть
        if not is_new and self.initial and name in ('employee', 'comment'):
            return FIELD_STATE.READONLY

        return super(BaseEmployeeForm, self).get_field_state(name)


class BaseHeadForm(TreeForm):
    """Базовая форма для шапки анкеты."""

    is_new = CharField(state=FIELD_STATE.READONLY)
    is_locked = BooleanField(state=FIELD_STATE.READONLY)
    event_type = CharField(state=FIELD_STATE.READONLY)

    purpose = ModelMultipleChoiceField(
        queryset=TripPurpose.objects.filter(intranet_status=1),
        empty_label=None,
        label=_('trip.label.purpose'),
        placeholder=_('trip.placeholder.purpose'),
        help_text=_('trip.help.purpose'),
        state=FIELD_STATE.REQUIRED,
    )

    comment = CharField(
        label=_('trip.label.comment'),
        help_text=_('trip.help.comment'))

    employee_list = TreeFormField(
        BaseEmployeeForm,
        label=_('trip.label.employee_list'),
        help_text=_('trip.help.employee_list'),
    )

    def get_is_locked(self, initial):
        return False

    def get_field_state(self, name):

        is_new = self.initial.get('is_new')

        if not is_new and name == 'comment':
            return FIELD_STATE.READONLY

        return super(BaseHeadForm, self).get_field_state(name)

    def clean_purpose(self, purpose_list):
        if len(purpose_list) > 2:
            self.set_error('purpose', _('trip.error.too_many_purposes'))
            return []

        return [p.id for p in purpose_list]

    def clean(self):
        cd = super(BaseHeadForm, self).clean()

        employee_list = cd.get('employee_list')

        if not employee_list:
            msg = _('trip.error.employee_list_is_empty')
            self.set_error('employee_list', msg)
        else:
            employees = [e.get('employee') for e in employee_list]

            if len(employees) != len(set(employees)):
                msg = _('trip.error.repeated_employee')
                self.set_error('employee_list', msg)

        return cd


class EventDateHeadForm(TreeForm):
    """Форма дат события."""

    event_date_from = DateField(
        label=_('trip.label.event_date_from'),
        help_text=_('trip.help.event_date_from'))
    event_date_to = DateField(
        label=_('trip.label.event_date_to'),
        help_text=_('trip.help.event_date_to'))

    def clean(self):

        cd = super(EventDateHeadForm, self).clean()

        event_date_from = cd.get('event_date_from')
        event_date_to = cd.get('event_date_to')

        if event_date_from or event_date_to:
            # Если хотябы одна из дат заполнена
            # должна быть заполнена и вторая
            if not event_date_from:
                msg = _('trip.error.event_date_from_and_event_date_to')
                self.set_error("event_date_from", msg)
            elif not event_date_to:
                msg = _('trip.error.event_date_from_and_event_date_to')
                self.set_error("event_date_to", msg)
            else:
                # Дата начала мероприятия должна быть меньше
                # даты окончания
                if event_date_from > event_date_to:
                    msg = _('trip.error.event_date_from_longer_event_date_to')
                    self.set_error("event_date_to", msg)

        return cd


class DateHeadForm(EventDateHeadForm):
    """Форма дат поездки в связке с датами события."""

    trip_date_from = DateField(
        label=_('trip.label.trip_date_from'),
        help_text=_('trip.help.trip_date_from'),
        state=FIELD_STATE.REQUIRED)
    trip_date_to = DateField(
        label=_('trip.label.trip_date_to'),
        help_text=_('trip.help.trip_date_to'),
        state=FIELD_STATE.REQUIRED)

    def __init__(self, *args, **kwargs):
        super(DateHeadForm, self).__init__(*args, **kwargs)
        if self.data and len(self.data['city_list']) > 1:
            self.data = update_dates(self.data)

    def clean(self):
        cd = super(DateHeadForm, self).clean()

        trip_date_from = cd.get('trip_date_from')
        trip_date_to = cd.get('trip_date_to')

        if trip_date_from and trip_date_to and trip_date_from > trip_date_to:
            msg = _('trip.error.trip_date_from_longer_trip_date_to')
            self.set_error("trip_date_to", msg)
        return cd


# Формы командировок
class TripCityForm(TreeForm):
    """
        Форма города в который организуется командировка.
        Этих форм может быть несколько на странице
    """

    is_return_route = BooleanField(
        default=False,
        label=_('trip.label.city-is_return_route'),
        help_text=_('trip.help.city-is_return_route'))
    city = CharField(
        label=_('trip.label.city-city'),
        help_text=_('trip.help.city-city'))
    country = CharField(
        label=_('trip.label.city-country'),
        help_text=_('trip.help.city-country'))
    transport = ChoiceField(
        label=_('trip.label.city-transport'),
        help_text=_('trip.help.city-transport'),
        empty_label=None,
        default=TRANSPORT.AIRCRAFT,
        choices=TRANSPORT,
        state=FIELD_STATE.REQUIRED)
    baggage = ChoiceField(
        label=_('trip.label.city-baggage'),
        help_text=_('trip.help.city-baggage'),
        empty_label=None,
        choices=BAGGAGE)
    fare = ChoiceField(
        label=_('trip.label.city-fare'),
        help_text=_('trip.help.city-fare'),
        empty_label=None,
        default=FARE.MOST_ECONOMICAL,
        choices=FARE)
    city_arrive_date_type = ChoiceField(
        label=_('trip.label.city-arrival-date-type'),
        help_text=_('trip.help.city-arrival-date-type'),
        empty_label=None,
        default=ARRIVE_DATE_TYPE.DEPARTURE,
        choices=ARRIVE_DATE_TYPE,
        state=FIELD_STATE.REQUIRED)
    departure_date = DateField(
        label=_('trip.label.city-departure_date'),
        help_text=_('trip.help.city-departure_date'),
        state=FIELD_STATE.REQUIRED)
    time_proposal = CharField(
        label=_('trip.label.city-time_proposal'),
        help_text=_('trip.help.city-time_proposal'))
    has_tickets = BooleanField(
        default=False,
        label=_('trip.label.city-has_tickets'),
        help_text=_('trip.help.city-has_tickets'))
    tickets_cost = IntegerField(
        label=_('trip.label.city-tickets_cost'),
        help_text=_('trip.help.city-tickets_cost'))
    tickets_cost_currency = CharField(
        label=_('trip.label.city-tickets_cost_currency'),
        help_text=_('trip.help.city-tickets_cost_currency'))
    car_rent = BooleanField(
        default=False,
        label=_('trip.label.city-car_rent'),
        help_text=_('trip.help.city-car_rent'))
    need_hotel = BooleanField(
        default=True,
        label=_('trip.label.city-need_hotel'),
        help_text=_('trip.help.city-need_hotel'))
    hotel = CharField(
        label=_('trip.label.city-hotel'),
        help_text=_('trip.help.city-hotel'))
    comment = CharField(
        label=_('trip.label.city-comment'),
        placeholder=_('trip.placeholder.city-comment'),
        help_text=_('trip.help.city-comment'))
    ready_to_upgrade = BooleanField(
        default=False,
        label=_('trip.label.city-ready_to_upgrade'),
        help_text=_('trip.help.city-ready_to_upgrade'))
    upgrade_comment = CharField(
        label=_('trip.label.city-upgrade_comment'),
        placeholder=_('trip.placeholder.city-upgrade_comment'),
        help_text=_('trip.help.city-upgrade_comment'))

    def get_field_state(self, name):
        is_new = self.parent_form.initial.get('is_new')

        readonly_fields = (
            'country',
            'city',
            'transport',
            'baggage',
            'fare',
            'car_rent',
            'need_hotel',
            'hotel',
            'has_tickets',
            'tickets_cost',
            'tickets_cost_currency',
            'comment',
            'ready_to_upgrade',
            'upgrade_comment',
        )

        # Если не новая и уже есть сохраненная часть
        if not is_new and self.initial and name in readonly_fields:
            return FIELD_STATE.READONLY

        if name in ('city', 'country'):
            cd = self.cleaned_data
            is_return_route = cd.get('is_return_route',
                                     self.initial.get('is_return_route'))
            if not is_return_route:
                return FIELD_STATE.REQUIRED

        if name in ('tickets_cost', 'tickets_cost_currency'):
            cd = self.cleaned_data
            has_tickets = cd.get('has_tickets', self.initial.get('has_tickets'))
            if has_tickets:
                return FIELD_STATE.REQUIRED

        if name == 'baggage':
            cd = self.cleaned_data
            is_aircraft = cd.get('transport', self.initial.get('transport')) == TRANSPORT.AIRCRAFT
            if is_aircraft:
                return FIELD_STATE.REQUIRED

        return super(TripCityForm, self).get_field_state(name)


class InterestedUserForm(TreeForm):
    """
        Форма email человека или рассылки
        на который надо отправить уведомление.
        Этих форм может быть несколько для каждого участника
    """
    employee = EmailField(
        label=_('trip.label.interested_user-employee'),
        state=FIELD_STATE.REQUIRED)


class TripEmployeeForm(BaseEmployeeForm):
    """
        Форма участника командировки.
        Этих форма может быть несколько на странице
    """

    employee_assignment = ValidChoiceField(
        label=_('trip.label.employee-employee_assignment'),
        help_text=_('trip.help.employee-employee_assignment'),
        empty_label=None,
        choices=EMPTY_CHOICES)
    passport_number = CharField(
        label=_('trip.label.employee-passport_number'),
        help_text=_('trip.help.employee-passport_number'),
        state=FIELD_STATE.REQUIRED)
    passport_name = CharField(
        label=_('trip.label.employee-passport_name'),
        help_text=_('trip.help.employee-passport_name'),
        state=FIELD_STATE.REQUIRED)
    compensation = ChoiceField(
        label=_('trip.label.employee-compensation'),
        help_text=_('trip.help.employee-compensation'),
        empty_label=None,
        default=COMPENSATION.MONEY,
        choices=COMPENSATION,
    )
    mobile_packages = NullBooleanField(
        label=_('trip.label.employee-mobile_packages'),
        help_text=_('trip.help.employee-mobile_packages'),
        state=FIELD_STATE.REQUIRED,
    )
    need_mobile_additional_packages = CharField(
        label=_('trip.label.employee-need_mobile_additional_packages'),
        placeholder=_(
            'trip.placeholder.employee-need_mobile_additional_packages'),
        help_text=_('trip.help.employee-need_mobile_additional_packages'))
    mobile_date_from = DateField(
        label=_('trip.label.mobile_date_from'),
        help_text=_('trip.help.mobile_date_from'))
    mobile_date_to = DateField()

    corporate_mobile_no = CharField(
        label=_('trip.label.corporate-mobile_no'),
        help_text=_('trip.help.corporate-mobile_no'),
    )
    need_taxi = NullBooleanField(
        label=_('trip.label.employee-need_taxi'),
        help_text=_('trip.help.employee-need_taxi'),
        state=FIELD_STATE.REQUIRED,
    )
    mobile_number_for_taxi = ValidChoiceField(
        label=_('trip.label.mobile-number-for-taxi'),
        help_text=_('trip.help.mobile-number-for-taxi'),
        empty_label=None,
        choices=EMPTY_CHOICES
    )
    need_copy_of_insurance = BooleanField(
        label=_('trip.label.employee-need_copy_of_insurance'),
        help_text=_('trip.help.employee-need_copy_of_insurance'))
    need_visa = BooleanField(
        label=_('trip.label.employee-need_visa'),
        help_text=_('trip.help.employee-need_visa'))

    has_holidays = BooleanField(
        label=_('trip.label.employee-has_holidays'),
        help_text=_('trip.help.employee-has_holidays'))
    holidays_comment = CharField(
        label=_('trip.label.employee-holidays_comment'),
        placeholder=_('trip.placeholder.employee-holidays_comment'),
        help_text=_('trip.help.employee-holidays_comment'))
    custom_dates = BooleanField(
        label=_('trip.label.employee-custom_dates'),
        help_text=_('trip.help.employee-custom_dates'))
    departure_date = DateField(
        label=_('trip.label.employee-departure_date'),
        help_text=_('trip.help.employee-departure_date'))
    return_date = DateField(
        label=_('trip.label.employee-return_date'),
        help_text=_('trip.help.employee-return_date'))
    transfer = CharField(
        label=_('trip.label.employee-transfer'),
        help_text=_('trip.help.employee-transfer'))
    is_private = BooleanField(
        label=_('trip.label.employee-is_private'),
        help_text=_('trip.help.employee-is_private'))
    trip_info = CharField(
        label=_('trip.label.employee-trip_info'),
        placeholder=_('trip.placeholder.employee-trip_info'),
        help_text=_('trip.help.employee-trip_info'))
    comment = CharField(
        label=_('trip.trip.label.employee-comment'),
        placeholder=_('trip.placeholder.employee-comment'),
        help_text=_('trip.trip.help.employee-comment'))

    interested_user_list = TreeFormField(
        InterestedUserForm,
        label=_('trip.label.employee-interested_user_list'),
        help_text=_('trip.help.employee-interested_user_list'))

    trip_issue_status = CharField(
        label=_('trip.label.employee-trip_issue_status'),
        state=FIELD_STATE.READONLY)

    def get_trip_issue_status(self, initial):
        presentation = get_status_presentation(
            data=initial,
            event_type='trip',
            is_employee=True
        )
        return presentation

    def get_field_state(self, name):
        is_new = self.parent_form.initial.get('is_new')

        check_fields = (
            'has_holidays',
            'holidays_comment',
            'mobile_date_from',
            'mobile_date_to',
            'passport_number',
            'passport_name',
            'need_copy_of_insurance',
            'need_visa',
            'mobile_packages',
            'need_mobile_additional_packages',
            'corporate_mobile_no',
            'compensation',
            'employee_assignment',
            'need_taxi',
            'mobile_number_for_taxi',
        )
        # Если не новая и уже есть сохраненная часть
        if not is_new and self.initial and name in check_fields:
            return FIELD_STATE.READONLY

        if name in ('corporate_mobile_no', 'need_mobile_additional_packages'):
            mobile_packages = self.cleaned_data.get(
                'mobile_packages',
                self.initial.get('mobile_packages')
            )
            if mobile_packages:
                return FIELD_STATE.REQUIRED

        return super(TripEmployeeForm, self).get_field_state(name)

    def clean_mobile_number_for_taxi(self, value):
        if not self.cleaned_data['need_taxi']:
            return ''

        if value.strip() == '':
            self.set_error(
                'mobile_number_for_taxi',
                _('trip.error.mobile_number_for_taxi_is_not_valid')
            )
            return ''

        ff = PhoneField(required=False)

        try:
            result = ff.clean(value)
        except ValidationError:
            self.set_error(
                'mobile_number_for_taxi',
                _('trip.error.mobile_number_for_taxi_is_not_valid')
            )
            result = ''

        return result

    def clean(self):
        cd = super(TripEmployeeForm, self).clean()

        departure_date = cd.get('departure_date')
        return_date = cd.get('return_date')

        if departure_date and return_date and departure_date > return_date:
            msg = _('trip.error.gap_date_from_longer_gap_date_to')
            self.set_error("return_date", msg)
        return cd


class TripHeadForm(BaseHeadForm, DateHeadForm):
    """
        Форма шапки организуемой командировки.
        Планируется что эта форма будет одна на странице.
    """

    def __init__(self, *args, **kwargs):
        super(TripHeadForm, self).__init__(*args, **kwargs)

        initial_trip_data = kwargs.get('initial')
        if initial_trip_data and initial_trip_data.get('purpose'):
            self.base_fields['purpose'].queryset |= TripPurpose.objects.filter(
                id__in=initial_trip_data.get('purpose')
            )

    objective = CharField(
        label=_('trip.label.objective'),
        placeholder=_('trip.placeholder.objective'),
        help_text=_('trip.help.objective'),
        state=FIELD_STATE.NORMAL,
    )
    receiver_side = CharField(
        label=_('trip.label.city-receiver_side'),
        placeholder=_('trip.placeholder.city-receiver_side'),
        help_text=_('trip.help.city-receiver_side'),
    )
    city_list = TreeFormField(
        TripCityForm,
        label=_('trip.label.city_list'),
        help_text=_('trip.help.city_list'),
    )
    employee_list = TreeFormField(
        TripEmployeeForm,
        label=_('trip.label.employee_list'),
        help_text=_('trip.help.employee_list'),
    )

    trip_issue_url = CharField(
        label=_('trip.label.trip_issue_url'),
        state=FIELD_STATE.READONLY)

    def get_trip_issue_url(self, initial):
        return get_issue_url(initial, 'trip_issue')

    trip_issue_status = CharField(
        label=_('trip.label.trip_issue_status'),
        state=FIELD_STATE.READONLY)

    def get_trip_issue_status(self, initial):

        presentation = get_status_presentation(
            data=initial,
            event_type='trip',
            is_employee=False
        )

        return presentation

    def get_field_state(self, name):
        is_new = self.initial.get('is_new')

        if not is_new and name in ('receiver_side', 'objective'):
            return FIELD_STATE.READONLY

        return super(TripHeadForm, self).get_field_state(name)

    def _clean_city_list_date(self, cd, city_list):

        def set_error(num, msg):
            place = self.errors.setdefault('city_list', {})
            place = place.setdefault('errors', [{} for c in city_list])
            errors = place[num].setdefault('departure_date', [])
            errors.append(msg)

        trip_date_from = cd.get('trip_date_from')
        trip_date_to = cd.get('trip_date_to')

        first_route_city = 1
        if trip_date_from and trip_date_to:
            previous_date = None
            for num, city in enumerate(city_list[first_route_city:]):
                departure_date = city.get('departure_date')

                if not departure_date:
                    continue

                if previous_date and previous_date > departure_date:
                    msg = _('trip.error.previous_date_longer_departure_date')
                    set_error(num + first_route_city, msg)

                previous_date = departure_date

    def clean(self):
        cd = super(TripHeadForm, self).clean()

        city_list = cd.get('city_list')

        if not city_list:
            msg = _('trip.error.city_list_is_empty')
            self.set_error('city_list', msg)
        else:
            if len(city_list) == 2:
                if get_return_city(city_list):
                    msg = _('trip.error.destination_city_not_specified')
                    self.set_error('city_list', msg)
                    return cd

                city = city_list[1].get('city')
                departure_city = get_departure_city(city_list)['city']

                if city:
                    city_name = re.sub(r'\s', '', city.lower())
                    dep_city = re.sub(r'\s', '', departure_city.lower())
                    if city_name == dep_city:
                        msg = _('trip.error.one_city_eq_departure_city')
                        self.set_error('city_list', msg)

            if len(city_list) == 1:
                msg = _('trip.error.wrong_city_list_data')
                self.set_error('city_list', msg)

            self._clean_city_list_date(cd, city_list)

        return cd


# Формы конференций.
class ConfPartHeadForm(TreeForm):
    event_name = CharField(
        label=_('trip.label.event_name'),
        help_text=_('trip.help.event_name'),
        state=FIELD_STATE.REQUIRED)
    event_cost = CharField(
        label=_('trip.label.event_cost'),
        help_text=_('trip.help.event_cost'),
        state=FIELD_STATE.REQUIRED)

    conf_issue_url = CharField(
        state=FIELD_STATE.READONLY)

    def get_conf_issue_url(self, initial):
        return get_issue_url(initial, 'conf_issue')

    conf_issue_status = CharField(
        label=_('trip.label.conf_issue_status'),
        state=FIELD_STATE.READONLY)

    def get_conf_issue_status(self, initial):

        presentation = get_status_presentation(
            data=initial,
            event_type='conf',
            is_employee=False
        )

        return presentation


class ConfPartEmployeeForm(TreeForm):
    event_role = ChoiceField(
        label=_('trip.label.employee-event_role'),
        help_text=_('trip.help.employee-event_role'),
        empty_label=None,
        default=EVENT_ROLE.LISTENER,
        choices=EVENT_ROLE.choices())

    conf_issue_status = CharField(
        label=_('conf.label.employee-conf_issue_status'),
        state=FIELD_STATE.READONLY)

    def get_conf_issue_status(self, initial):

        presentation = get_status_presentation(
            data=initial,
            event_type='conf',
            is_employee=True
        )

        return presentation


class ConfEmployeeForm(BaseEmployeeForm, ConfPartEmployeeForm):
    comment = CharField(
        label=_('trip.conf.label.employee-comment'),
        help_text=_('trip.conf.help.employee-comment'))


class ConfHeadForm(ConfPartHeadForm, BaseHeadForm, EventDateHeadForm):

    objective = CharField(
        label=_('trip.label.objective_conf'),
        placeholder=_('trip.placeholder.objective_conf'),
        help_text=_('trip.help.objective_conf'),
        state=FIELD_STATE.REQUIRED,
    )
    employee_list = TreeFormField(
        ConfEmployeeForm,
        label=_('trip.label.employee_list'),
        help_text=_('trip.help.employee_list'),
    )

    def get_field_state(self, name):
        if name in ('event_date_from', 'event_date_to'):
            return FIELD_STATE.REQUIRED

        is_new = self.initial.get('is_new')

        if not is_new and name == 'objective':
            return FIELD_STATE.READONLY

        return super(ConfHeadForm, self).get_field_state(name)


# Смешаный сценари командировка и конференция
class TripConfEmployeeForm(TripEmployeeForm, ConfPartEmployeeForm):
    comment = CharField(
        label=_('trip.trip_conf.label.employee-comment'),
        help_text=_('trip.trip_conf.help.employee-comment'),
        placeholder=_('trip.placeholder.employee-comment'),
    )


class TripConfHeadForm(TripHeadForm, ConfPartHeadForm):
    employee_list = TreeFormField(
        TripConfEmployeeForm,
        label=_('trip.label.employee_list'),
        help_text=_('trip.help.employee_list'),
    )

    def get_field_state(self, name):
        if name in ('event_date_from', 'event_date_to'):
            return FIELD_STATE.REQUIRED
        return super(TripConfHeadForm, self).get_field_state(name)


# Другие вспомогательные формы.
class GapDataForm(forms.Form):
    left_edge = forms.DateField(input_formats=DATE_INPUT_FORMATS)
    right_edge = forms.DateField(input_formats=DATE_INPUT_FORMATS)
    employee = forms.ModelChoiceField(Staff.objects.active(),
                                      to_field_name='login')


class TripHistoryFilterForm(forms.Form):
    user = forms.ModelChoiceField(
        Staff.objects.all(),
        to_field_name='login',
        required=False
    )
    page = forms.IntegerField(
        min_value=1,
        required=False,
    )
    limit = forms.IntegerField(required=False)
    role = forms.ChoiceField(EVENT_ROLES, required=False)
    event_type = forms.ChoiceField(EVENT_TYPE_EXTENDED, required=False)


class OldTripsFilterForm(forms.Form):
    event_type = forms.ChoiceField(EVENT_TYPE_EXTENDED, required=False)
    participants = forms.CharField(required=True)
    date_from = forms.DateField(required=True)
    date_to = forms.DateField(required=True)

    def clean_participants(self) -> List[str]:
        """Принимает логины, отдаёт id"""
        logins = self.cleaned_data['participants']
        if not logins:
            return []
        if self.requester.has_perm('django_intranet_stuff.can_view_trip_history'):
            return list(
                Staff.objects
                .filter(login__in=logins.split(','))
                .values_list('id', flat=True)
            )
        return [self.requester.get_profile().id]

    @classmethod
    def for_user(cls, data, user):
        form_instance = cls(data)
        form_instance.requester = user
        return form_instance
