from typing import List, Tuple
from datetime import datetime, time, date
from .mongo_models import mongo

STAFF_TRIPS_QUERIES = {
    'author': lambda person_id: {'author': person_id},
    'participant': lambda person_id: {'employee_list.employee': person_id},
    'all': lambda person_id: {
        '$or' [
            STAFF_TRIPS_QUERIES['author'](person_id),
            STAFF_TRIPS_QUERIES['participant'](person_id),
        ]
    },
}


def APPEND_PERSON_QUERY(query, person_id):
    return {
        '$and': [
            query,
            {'$or': [
                {'author': person_id},
                {'employee_list.employee': person_id},
            ]
            }
        ]
    }


EVENT_TYPES = ['trip', 'conf', 'trip_conf']
REQUIRED_FIELDS = [
    'city_list',
    'trip_date_from',
    'trip_date_to',
    'event_date_from',
    'event_date_to',
    'author',
    'event_type',
    'employee_list',
    'event_name',
    'objective',
    'uuid',
    'conf_issue',
    'trip_issue',
]


def get_staff_trip_history(target_staff_id, role='all', event_type=None, **kw):
    """
    Возвращает список заявок сотрудника target_staff_id, с фильтром по
    указанныи параметрам.
    дополнительный, но необязательный фильтр:  kw['user'] - id пользователя,
    который должен быть автором либо участником поездки.
    """
    additional_user = kw.get('user')

    query = STAFF_TRIPS_QUERIES[role](str(target_staff_id))
    if event_type in EVENT_TYPES:
        query['event_type'] = event_type
    if additional_user:
        query = APPEND_PERSON_QUERY(query, additional_user)

    all_records = sorted(
        mongo.db.trip_questionary.find(query,),
        key=lambda x: x.get('trip_date_from') or x.get('event_date_from'),
        reverse=True,
    )

    for record in all_records:
        record['employee_list_ids'] = [
            e['employee'] for e in record['employee_list']
        ]

    return all_records


def get_trip_list(
        participants: List[int],
        date_from: date,
        date_to: date,
        event_type=None,
        limit=100,
        offset=0
) -> Tuple[List[dict], int]:

    query = {'employee_list.employee': {'$in': [str(p) for p in participants]}}
    if event_type in EVENT_TYPES:
        query['event_type'] = event_type
    date_from = datetime.combine(date_from, time())
    date_to = datetime.combine(date_to, time())

    query['$or'] = [
        {
            'trip_date_to': {'$gte': date_from},
            'trip_date_from': {'$lte': date_to},
        },
        {
            'event_date_to': {'$gte': date_from},
            'event_date_from': {'$lte': date_to},
        },
    ]
    all_records = sorted(
        mongo.db.trip_questionary.find(query).sort([('creation_time', -1)]).skip(offset).limit(limit),
        key=lambda x: x.get('trip_date_from') or x.get('event_date_from'),
        reverse=True,
    )
    count = mongo.db.trip_questionary.find(query).count()
    for record in all_records:
        record['_id'] = str(record.pop('_id'))
    return all_records, count


def get_trip_list_by_uids(trip_uuids: List[str]) -> List[dict]:
    query = {'uuid': {'$in': trip_uuids}}
    all_records = list(mongo.db.trip_questionary.find(query))
    for record in all_records:
        record['_id'] = str(record.pop('_id'))
    return all_records
