import copy

from django import forms
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext
from django.utils.encoding import smart_text
from django.db.models import Model
from collections import OrderedDict
from django.utils.translation import ugettext_lazy as _

from staff.lib.utils.ordered_choices import StaffChoices

from staff.lib.forms.fields import (
    MulticModelChoiceField as DjMulticModelChoiceField
)


__all__ = (
    'FIELD_STATE',
    'TreeForm', 'TreeFormField', 'ModelChoiceField', 'MulticModelChoiceField',
    'CharField', 'DateField', 'EmailField', 'BooleanField', 'ChoiceField',
    'ValidChoiceField', 'NullBooleanField',
)

# 'IntegerField','TimeField','DateTimeField', 'TimeField','RegexField',
# 'FileField', 'ImageField', 'URLField','NullBooleanField',
# 'MultipleChoiceField',
# 'ComboField', 'MultiValueField', 'FloatField', 'DecimalField',
# 'SplitDateTimeField', 'GenericIPAddressField',
# 'FilePathField',
# 'SlugField', 'TypedChoiceField', 'TypedMultipleChoiceField'

FIELD_STATE = StaffChoices(
    REQUIRED=('required', 'required'),
    NORMAL=('normal', 'normal'),
    READONLY=('readonly', 'readonly'),
)


class Field(object):

    creation_counter = 0

    def __init__(self, label='', state=FIELD_STATE.NORMAL, default='',
                 empty_label="---------", help_text='', placeholder='', *args, **kwargs):

        super(Field, self).__init__()

        self.creation_counter = Field.creation_counter
        Field.creation_counter += 1

        self.label = label
        self.state = state
        self.empty_label = empty_label
        self.help_text = help_text
        self.args = args
        self.kwargs = kwargs
        self.default = default
        self.placeholder = placeholder

    def set_form_and_name(self, form, name):
        self.form = form
        self.name = name

    def clean(self, new_value, old_value, required):
        result, has_error = None, False
        try:
            dj_field = self.get_dj_field(required)
            result = dj_field.clean(new_value)
        except forms.ValidationError as e:
            errors = e.messages
            for error_msg in errors:
                self.set_error(error_msg)
            has_error = True

        return result, has_error

    def get_dj_field(self, required=False):
        raise NotImplementedError()

    def as_dict(self, name, value, state):
        def smart_ugettext(s):
            return ugettext(s).strip() if s else ''

        field_dict = {
            'placeholder': smart_ugettext(self.placeholder),
            'help_text': smart_ugettext(self.help_text),
            'label': smart_ugettext(self.label),
            'name': name,
            'value': value,
            'required': state == FIELD_STATE.REQUIRED,
            'readonly': state == FIELD_STATE.READONLY,
            'type': self.__class__.__name__,
        }

        return field_dict

    def set_error(self, error_msg):
        error_list = self.form._errors.setdefault(self.name, [])
        error_list.append(error_msg)


class BooleanField(Field):
    def __init__(self, default=False, *args, **kwargs):
        kwargs['default'] = default
        super(BooleanField, self).__init__(*args, **kwargs)

    def get_dj_field(self, required=False):
        return forms.BooleanField(required=required, *self.args, **self.kwargs)


class ValidatedNullBooleanField(forms.NullBooleanField):
    def validate(self, value):
        if value is None and self.required:
            raise ValidationError(self.error_messages['required'])


class NullBooleanField(BooleanField):
    def __init__(self, default=None, *args, **kwargs):
        super(NullBooleanField, self).__init__(default, *args, **kwargs)

    def get_dj_field(self, required=False):
        return ValidatedNullBooleanField(required=required, *self.args,
                                         **self.kwargs)


class DateField(Field):
    def get_dj_field(self, required=False):
        return forms.DateField(required=required, *self.args, **self.kwargs)


class CharField(Field):
    def get_dj_field(self, required=False):
        return forms.CharField(required=required, *self.args, **self.kwargs)


class IntegerField(Field):
    def get_dj_field(self, required=False):
        return forms.IntegerField(required=required, *self.args, **self.kwargs)


class EmailField(Field):
    def get_dj_field(self, required=False):
        return forms.EmailField(required=required, *self.args, **self.kwargs)


class ChoiceField(Field):
    def __init__(self, choices=(), *args, **kwargs):
        super(ChoiceField, self).__init__(*args, **kwargs)
        self.choices = choices

    def get_dj_field(self, required=False):
        return forms.ChoiceField(required=required,
                                 choices=self.choices,
                                 *self.args,
                                 **self.kwargs)

    def as_dict(self, *args, **kwargs):
        field_dict = super(ChoiceField, self).as_dict(*args, **kwargs)
        choices = []
        if self.empty_label is not None:
            choices.append({'value': '', 'name': self.empty_label})
        choices += [{
            'value': v,
            'name': ugettext(n)
        } for v, n in self.choices]
        field_dict['choices'] = choices
        return field_dict


class ValidDjChoiceField(forms.ChoiceField):
    def __init__(self, *args, **kwargs):
        super(ValidDjChoiceField, self).__init__(*args, **kwargs)

    def validate(self, value):
        pass


class ValidChoiceField(ChoiceField):

    def get_dj_field(self, required=False):
        return ValidDjChoiceField(required=required,
                                  choices=self.choices,
                                  *self.args,
                                  **self.kwargs)

    def as_dict(self, *args, **kwargs):
        field_dict = super(ValidChoiceField, self).as_dict(*args, **kwargs)
        choices = []
        if self.empty_label is not None:
            choices.append({'value': '', 'name': self.empty_label})
        choices += [{
            'value': v,
            'name': ugettext(n)
        } for v, n in self.choices]
        field_dict['choices'] = choices
        return field_dict


class ModelChoiceField(Field):
    def __init__(self, queryset, *args, **kwargs):
        super(ModelChoiceField, self).__init__(*args, **kwargs)
        self.queryset = queryset

    def get_dj_field(self, required=False):
        return forms.ModelChoiceField(
            queryset=self.queryset,
            required=required,
            *self.args,
            **self.kwargs
        )

    def as_dict(self, *args, **kwargs):
        field_dict = super(ModelChoiceField, self).as_dict(*args, **kwargs)

        def choice_gen():
            if self.empty_label is not None:
                yield '', self.empty_label
            for obj in self.queryset.all():
                yield obj.pk, self.choice_name_from_instance(obj)

        choices = [{'value': v, 'name': n} for v, n in choice_gen()]
        field_dict['choices'] = choices

        if isinstance(field_dict['value'], Model):
            field_dict['value'] = field_dict['value'].pk
        return field_dict

    def choice_name_from_instance(self, obj):
        return smart_text(obj)


class ModelMultipleChoiceField(ModelChoiceField):

    def __init__(self, queryset, *args, **kwargs):
        super(ModelMultipleChoiceField, self).__init__(queryset, *args, **kwargs)
        self.queryset = queryset

    def get_dj_field(self, required=False):
        return forms.ModelMultipleChoiceField(
            queryset=self.queryset,
            required=required,
            *self.args,
            **self.kwargs
        )

    def as_dict(self, *args, **kwargs):
        field_dict = super(ModelMultipleChoiceField, self).as_dict(*args, **kwargs)

        def choice_gen():
            if self.empty_label is not None:
                yield '', self.empty_label
            for obj in self.queryset.all():
                yield obj.pk, self.choice_name_from_instance(obj)

        choices = [{'value': v, 'name': n} for v, n in choice_gen()]
        field_dict['choices'] = choices

        field_dict['value'] = [
            val.pk if isinstance(val, Model) else val
            for val in field_dict['value']
        ]

        return field_dict


class MulticModelChoiceField(Field):
    def __init__(self, queryset, *args, **kwargs):
        super(MulticModelChoiceField, self).__init__(*args, **kwargs)
        self.queryset = queryset

    def get_dj_field(self, required=False):
        return DjMulticModelChoiceField(
            queryset=self.queryset,
            required=required,
            *self.args, **self.kwargs)

    def as_dict(self, name, value, state):
        field_dict = super(MulticModelChoiceField, self).as_dict(name, value, state)
        dj_field = self.get_dj_field()
        presentation = dj_field.presentation_function(value) if value else ''
        field_dict['multic_values'] = {
            'type': dj_field.multic_type,
            'value': presentation,
        }
        if isinstance(field_dict['value'], Model):
            field_dict['value'] = field_dict['value'].pk
        return field_dict


class TreeFormField(Field):
    def __init__(self, tree_form_cls, default=None, *args, **kwargs):
        self.tree_form_cls = tree_form_cls
        default = [] if default is None else default
        super(TreeFormField, self).__init__(default=default, *args, **kwargs)

    def clean(self, new_value, old_value, required):

        if not isinstance(new_value, list):
            self.set_error(_('Enter a valid value.'))
            self.set_error(_('Should be a list.'))
            return [], True

        def value_gen():
            for num, new_value_item in enumerate(new_value):
                try:
                    old_value_item = old_value[num]
                except IndexError:
                    old_value_item = {}
                yield new_value_item, old_value_item

        tree_forms = [
            self.tree_form_cls(data=nv, initial=ov, parent_form=self.form)
            for nv, ov in value_gen()
        ]

        errors = [b.errors for b in tree_forms]
        has_error = any(errors)
        if has_error:
            self._set_error_list(errors)

        cleaned_data = [b.cleaned_data for b in tree_forms]

        return cleaned_data, has_error

    def as_dict(self, name, value, state):
        field_dict = super(TreeFormField, self).as_dict(name, value, state)
        tree_forms = (self.tree_form_cls(parent_form=self.form, initial=v)
                      for v in value)
        field_dict['value'] = [b.as_dict() for b in tree_forms]
        return field_dict

    def _set_error_list(self, error_list):
        error_place = self.form._errors.setdefault(self.name, {})
        error_place['errors'] = error_list

    def set_error(self, error_msg):
        error_place = self.form._errors.setdefault(self.name, {})
        error_place = error_place.setdefault('all', [])
        error_place.append(error_msg)


class TreeFormMetaclass(type):
    def __new__(mcs, name, bases, attrs):
        fields = []
        for field_name, obj in list(attrs.items()):
            if isinstance(obj, Field):
                fields.append((field_name, attrs.pop(field_name)))
        fields.sort(key=lambda x: x[1].creation_counter)

        for base in bases[::-1]:
            if hasattr(base, 'base_fields'):
                fields = list(base.base_fields.items()) + fields

        attrs['base_fields'] = OrderedDict(fields)

        new_class = super(TreeFormMetaclass,
                          mcs).__new__(mcs, name, bases, attrs)

        return new_class


class BaseTreeForm(object):

    def __init__(self, data=None, initial=None, parent_form=None):
        # print self.__class__.__name__, parent_form
        self.data = data or {}
        self.initial = initial or {}
        self.parent_form = parent_form

        self._errors = None
        self.cleaned_data = {}

        self.fields = copy.deepcopy(self.base_fields)

        for name, field in self.fields.items():
            field.set_form_and_name(self, name)

        self.fields_state = self.prepare_fields_state()

    def prepare_fields_state(self):
        _fields = self.fields.items()
        return dict((name, f.state) for name, f in _fields)

    def _clean_fields(self):
        self._errors = {}

        for name, field in self.fields.items():
            field_state = self.get_field_state(name)

            old_value = self._get_initial_value(name, field)
            if field_state == FIELD_STATE.READONLY:
                if name in self.initial:
                    self.cleaned_data[name] = old_value
                continue

            new_value = self.data.get(name, field.default)

            value, has_error = field.clean(
                new_value=new_value,
                old_value=old_value,
                required=(field_state == FIELD_STATE.REQUIRED)
            )

            self.cleaned_data[name] = value

            if not has_error:
                if hasattr(self, 'clean_%s' % name):
                    value = getattr(self, 'clean_%s' % name)(value)
                    self.cleaned_data[name] = value

    def set_error(self, field_name, error_msg):
        field = self.fields[field_name]
        field.set_error(error_msg)

    def _get_errors(self):
        if self._errors is None:
            self._clean_fields()
            self.cleaned_data = self.clean()
        return self._errors

    def _get_initial_value(self, name, field):
        getter = getattr(self, 'get_{name}'.format(name=name), None)

        if getter is not None:
            value = getter(self.initial)
        else:
            value = self.initial.get(name, field.default)

        return value

    errors = property(_get_errors)

    def is_valid(self):
        return not bool(self.errors)

    def clean(self):
        return self.cleaned_data

    def get_field_state(self, name):
        return self.fields_state[name]

    def as_dict(self):
        result = {}
        for name, field in self.fields.items():
            value = self._get_initial_value(name, field)
            state = self.get_field_state(name)
            result[name] = field.as_dict(name, value, state)
        return result


class TreeForm(BaseTreeForm, metaclass=TreeFormMetaclass):
    pass
