import logging
import json
from typing import Dict, Any
from datetime import date, timedelta

from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods, require_POST
from staff.lib.decorators import responding_json, auth_by_tvm_only
from staff.lib.decorators import paginated
from staff.trip_questionary.forms import OldTripsFilterForm
from staff.trip_questionary.views import get_assignment_choices, save_trip as save
from staff.trip_questionary.sorted import get_trip_list, get_trip_list_by_uids
from staff.users.models import User

logger = logging.getLogger(__name__)


@csrf_exempt
@auth_by_tvm_only(['trip'])
@responding_json
@require_http_methods(["POST"])
def trip_list_raw(request) -> Dict[str, Any]:
    try:
        data = json.loads(request.body)
        trip_uuids = data['trip_uuids']
    except ValueError:
        logger.exception('Wrong JSON: %s', request.body)
        return {'error': 'wrong input json'}, 400
    except KeyError:
        logger.exception('No `trip_uuids` key', str(data.keys()))
        return {'error': 'no `trip_uuids` key'}, 400

    raw_trips_data = get_trip_list_by_uids(trip_uuids)
    return raw_trips_data


@require_http_methods(['GET'])
@responding_json
@paginated
def trip_list_filter(request, paginator):
    requester = request.user
    date_to = date.today()
    date_from = date_to - timedelta(days=365 * 30)
    params = {
        'event_type': 'all',
        'date_from': date_from.isoformat(),
        'date_to': date_to.isoformat(),
        'participants': requester.get_profile().login,
    }
    params.update(request.GET.dict())
    form = OldTripsFilterForm.for_user(data=params, user=requester)
    if not form.is_valid():
        return {'errors': form.errors}

    paginator.result, paginator.total = get_trip_list(
        limit=paginator.limit,
        offset=paginator.offset,
        **form.cleaned_data,
    )
    return paginator.as_dict()


@csrf_exempt
@auth_by_tvm_only(['trip'])
@responding_json
@require_POST
def save_trip(request):
    """Ручка создания и редактирования командировок из сервиса Trip"""
    login = request.GET.get('login', request.POST.get('login'))
    try:
        request.user = User.objects.get(username=login)
    except User.DoesNotExist:
        return {'errors': 'User `%s` not found in database' % login}, 404

    return save(request)


@auth_by_tvm_only(['trip'])
@responding_json
@require_http_methods(["GET"])
def assignments_by_logins(request) -> Dict[str, Any]:
    logins = request.GET.get('logins', '').lower().split(',')
    return get_assignment_choices(logins)
