import logging
from datetime import date
from json import loads

import yenv

from collections import defaultdict

from phonenumbers import (
    format_number,
    PhoneNumberFormat,
    parse,
    NumberParseException,
)

from django.core.context_processors import csrf
from django.core.paginator import Paginator, InvalidPage, EmptyPage
from django.core.urlresolvers import reverse
from django.http import (
    Http404,
    HttpResponseForbidden,
    HttpResponseRedirect,
    HttpResponseBadRequest,
)
from django.shortcuts import render
from django.utils.translation import ugettext_lazy as _
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods, require_POST

from staff.lib.decorators import responding_json, front_json, require_permission
from staff.lib.lock import lock_manager
from staff.lib.log import log_context
from staff.oebs.utils import get_oebs_assignments_by_person_logins
from staff.person.models import Staff, StaffPhone, PHONE_KIND

from staff.trip_questionary.utils import get_assignment_line
from staff.trip_questionary.controller.context import make_cities_chain, get_issue_url
from staff.trip_questionary.controller.operations.registry import delay_diff_operations
from staff.trip_questionary.forms import (
    TripConfHeadForm,
    TripHeadForm,
    ConfHeadForm,
    GapDataForm,
    TripHistoryFilterForm,
    BAGGAGE,
)
from staff.trip_questionary.models import TripQuestionaryCollection, EVENT_TYPE
from staff.trip_questionary.monitor import get_wierd_trips
from staff.trip_questionary.sorted import get_staff_trip_history
from staff.trip_questionary.tasks import trip_async_operator

logger = logging.getLogger(__name__)


def _get_initial(request):
    employee = request.user.get_profile()

    today = date.today().isoformat()

    default_location = {
        'city': employee.office.city.i_name,
        'country': employee.office.city.country.i_name,
        'departure_date': today,
        'baggage': BAGGAGE.HAND,
    }

    initial = {
        'employee_list': [{}],
        'city_list': [
            default_location,
            {},
            {
                'city': default_location['city'],
                'is_return_route': True
            }
        ],
    }

    if employee.login in ('may', 'volozh', 'dmirain'):
        initial['employee_list'][0]['is_private'] = True

    old_trip_uuid = request.POST.get('uuid', request.GET.get('uuid'))

    if 'employee' in request.POST:
        gap_data = request.POST
    elif 'employee' in request.GET:
        gap_data = request.GET
    else:
        gap_data = None

    if old_trip_uuid:
        old_trip = TripQuestionaryCollection().get(uuid=old_trip_uuid)
        if old_trip.is_new:
            raise Http404()

        initial = old_trip.data

        clerable_fields = (
            'trip_date_to',
            'trip_date_from',
            'event_date_from',
            'event_date_to',
            'has_holidays',
            'holidays_comment',
            'custom_dates',
            ('city_list', 'departure_date'),
            ('employee_list', 'departure_date'),
            ('employee_list', 'return_date'),
            ('employee_list', 'mobile_date_from'),
            ('employee_list', 'mobile_date_to'),
        )

        def clear_by_name(data, path, del_name):
            if path:
                path = list(path)
                name = path.pop()
                low_data = data.get(name)
                if low_data:
                    if isinstance(low_data, (tuple, list)):
                        for low_data_item in low_data:
                            clear_by_name(low_data_item, path, del_name)
                    else:
                        clear_by_name(low_data, path, del_name)
            elif del_name in data:
                del data[del_name]

        def clear_fields(initial, fields):

            for field in fields:
                if isinstance(field, (tuple, list)):
                    del_name = field[-1]
                    path = field[:-1]
                else:
                    del_name = field
                    path = []

                clear_by_name(initial, path, del_name)

        clear_fields(initial, old_trip.technical_fields)
        clear_fields(initial, clerable_fields)

        city_list = initial.get('city_list')
        if city_list:
            city_list[0]['departure_date'] = today

        return initial

    elif gap_data:
        prefill_form = GapDataForm(gap_data)
        assert prefill_form.is_valid()

        trip_date_from = prefill_form.cleaned_data['left_edge']
        trip_date_to = prefill_form.cleaned_data['right_edge']

        initial['trip_date_from'] = trip_date_from
        initial['trip_date_to'] = trip_date_to

        initial['event_date_from'] = trip_date_from
        initial['event_date_to'] = trip_date_to

        initial['employee_list'][0].update({
            'employee': prefill_form.cleaned_data['employee'],
        })

        departure_city = {
            'city': employee.office.city.name,
            'country': employee.office.city.country.name,
            'departure_date': trip_date_to,
            'is_return_route': False,
        }
        return_city = departure_city.copy()
        return_city.update({'is_return_route': True})
        first_city = {
            'departure_date': trip_date_from,
            'is_return_route': False,
        }

        initial['city_list'][0].update(departure_city)
        initial['city_list'][1].update(first_city)
        initial['city_list'][-1].update(return_city)

    return initial


def _access(request, trip):
    if trip.is_new:
        return True

    user = request.user
    if user.has_perm('django_intranet_stuff.can_edit_trip'):
        return True

    employee = user.get_profile()
    all_participant = [e['employee'] for e in trip.data['employee_list']]
    all_participant.append(trip.data.get('author'))
    if employee in all_participant:
        return True

    return False


def _get_form_class(event_type):
    forms = {
        EVENT_TYPE.TRIP: TripHeadForm,
        EVENT_TYPE.CONF: ConfHeadForm,
        EVENT_TYPE.TRIP_CONF: TripConfHeadForm,
    }
    return forms[event_type]


def _get_trip(trip_uuid, event_type, author):
    if trip_uuid is None:
        trip = TripQuestionaryCollection().new(author=author)
        trip.data['event_type'] = event_type
    else:
        trip = TripQuestionaryCollection().get(uuid=trip_uuid)
        if trip.is_new:
            raise Http404
        event_type = trip.data['event_type']
    return trip, event_type


@csrf_exempt
@require_http_methods(['GET', 'POST'])
@front_json
def edit(request):
    author = request.user.get_profile()
    trip_uuid = request.GET.get('trip_uuid', request.POST.get('trip_uuid'))
    event_type = request.GET.get('event_type', request.POST.get('event_type'))

    if trip_uuid is None and event_type not in ['trip', 'trip_conf', 'conf']:
        return HttpResponseBadRequest('trip_uuid or event_type is required')

    trip, event_type = _get_trip(trip_uuid, event_type, author)

    if not _access(request, trip):
        return HttpResponseForbidden(
            _('trip.access_deny.not_a_participant')
        )

    if trip.is_new:
        trip.data.update(_get_initial(request))

    trip_form = _get_form_class(event_type)(initial=trip.data)

    content = trip_form.as_dict()
    employee_list = content['employee_list']['value']

    person_ids = [
        emp['employee']['value']
        for emp in employee_list
        if emp['employee']['value']
    ]
    if person_ids:
        person_logins = dict(
            Staff.objects
            .filter(id__in=person_ids)
            .values_list('id', 'login')
        )

        assignment_choices = get_assignment_choices(list(person_logins.values()))
        phone_choices = _get_phones_choices(list(person_logins.values()))

        for employee_dict in employee_list:
            person_id = employee_dict['employee']['value']
            assignment_choices_data = assignment_choices.get(person_logins[person_id], [])
            phones_choices_data = phone_choices.get(person_logins[person_id], [])

            if assignment_choices_data and 'employee_assignment' in employee_dict:
                employee_dict['employee_assignment'].update(assignment_choices_data)

            if event_type != 'conf':
                mob_no_field = employee_dict['mobile_number_for_taxi']
                if phones_choices_data['choices']:
                    mob_no_field.update(phones_choices_data)
                    mob_no_field['value'] = mob_no_field['value'] or phones_choices_data['choices'][0]['value']

    return {
        'content': content,
        'csrf_token': csrf(request)['csrf_token'],
    }


def save_trip(request):
    author = request.user.get_profile()
    trip_uuid = request.GET.get('trip_uuid', request.POST.get('trip_uuid'))
    event_type = request.GET.get('event_type', request.POST.get('event_type'))

    if trip_uuid is None and event_type not in ['trip', 'trip_conf', 'conf']:
        return HttpResponseBadRequest('trip_uuid or event_type is required')

    trip, event_type = _get_trip(trip_uuid, event_type, author)

    if not _access(request, trip):
        return HttpResponseForbidden(
            _('trip.access_deny.not_a_participant')
        )

    initial = trip.data

    if 'content' in request.POST:
        data = request.POST['content']
    else:
        data = request.read()

    try:
        data = loads(data)
    except ValueError as e:
        return HttpResponseBadRequest(e)

    form = _get_form_class(event_type)(data=data, initial=initial)

    if form.is_valid():
        with log_context(trip_uuid=trip.uuid):
            trip.update(form.cleaned_data)

            delay_diff_operations(trip=trip)

            with lock_manager.lock(trip.lock_name, block=True):
                trip.save()

            method = trip_async_operator
            if yenv.type != 'development':
                method = method.delay
            method(trip_uuid=trip.uuid)

    else:
        return {
            'errors': [_('trip.error.save')],
            'content': form.errors,
        }

    kwargs = {'trip_uuid': trip.uuid, 'event_type': trip.data['event_type']}
    trip_url = '/trip/?trip_uuid=%(trip_uuid)s&event_type=%(event_type)s' % kwargs

    return {
        'messages': [_('trip.message.save-complite')],
        'content': {
            'trip_uuid': trip.uuid,
            'url': trip_url,
        },
    }


@require_POST
@front_json
def save(request):
    return save_trip(request)


def _get_trip_data(trip_data, staff_id):

    def get_dates(trip_data):
        _from = _to = ''
        if trip_data.get('trip_date_from'):
            _from = trip_data.get('trip_date_from')
            _to = trip_data.get('trip_date_to')

        if trip_data.get('event_date_from'):
            _from = trip_data.get('event_date_from')
            _to = trip_data.get('event_date_to')
        return {
            'from': _from.isoformat(),
            'to': _to.isoformat(),
        }
    staff_id = str(staff_id)

    def issue_url_type(data):
        return 'conf_issue' if data['event_type'] == 'conf' else 'trip_issue'

    is_author = trip_data['author'] == staff_id

    employee_data = trip_data
    for emp_data in trip_data['employee_list']:
        if emp_data['employee'] == staff_id:
            employee_data = emp_data

    def issue_url(trip_data, employee_data, meta_issue=True):
        data = trip_data if meta_issue else employee_data
        return get_issue_url(data, issue_url_type(trip_data))

    def issue_key(trip_data, employee_data, is_author):
        data = trip_data if is_author else employee_data
        return data.get(issue_url_type(trip_data), {}).get('key')

    return {
        'is_author': is_author,
        'is_participant': any([e['employee'] == staff_id
                               for e in trip_data['employee_list']]),
        'event_type': trip_data['event_type'],
        'dates': get_dates(trip_data),
        'city_list': make_cities_chain(trip_data),
        'caption': (trip_data.get('event_name') or
                    trip_data.get('objective', '')),
        'participants': list(
            Staff.objects.values_list('login', flat=True)
            .filter(id__in=trip_data['employee_list_ids'])
        ),
        'uuid': trip_data['uuid'],
        'issue_url': issue_url(trip_data, employee_data, meta_issue=is_author),
        'issue_key': issue_key(trip_data, employee_data, is_author=is_author),
    }


@require_http_methods(['GET'])
@front_json
def trip_history(request):
    params = {
        'page': 1,
        'limit': 8,
        'role': 'all',
        'event_type': 'all',
        'user': None,
    }
    params.update(request.GET)
    form = TripHistoryFilterForm(params)
    if not form.is_valid():
        return {'errors': form.errors}

    def _get_params(params, request):
        result = params
        requester = request.user
        has_perm = requester.has_perm(
            'django_intranet_stuff.can_view_trip_history'
        )

        result['target_staff_id'] = (
            result['user'].id if result['user'] else requester.get_profile().id
        )
        if not has_perm:
            if result['user']:
                result['user'] = requester.get_profile().id
        else:
            del result['user']

        return result

    params = _get_params(form.cleaned_data, request)

    staff_trips = get_staff_trip_history(**params)
    paginator = Paginator(staff_trips, params['limit'])

    try:
        trip_list = paginator.page(params['page'])
    except (EmptyPage, InvalidPage):
        params['page'] = paginator.num_pages
        trip_list = paginator.page(params['page'])

    trip_list.object_list = list(trip_list.object_list)

    return {
        'trip_list': [
            _get_trip_data(trip, params['target_staff_id'])
            for trip in trip_list
        ],
        'page': params['page'],
        'pages': paginator.num_pages,
        'params': {
            'role': params['role'],
            'event_type': params['event_type'] or 'all'
        },
    }


def get_assignment_choices(logins):
    empty_choice = {
        'name': "Назначение сотрудника в ОЕБС не найдено",
        'value': -999,
    }
    assignments = get_oebs_assignments_by_person_logins(logins)
    choices_dict = defaultdict(lambda: {})

    for login in logins:
        choices_dict[login]['choices'] = [empty_choice]

    for login, login_assignments in assignments.items():
        choices_dict[login]['choices'] = [
            {
                'name': get_assignment_line(assignment),
                'value': assignment['assignmentID']
            }
            for assignment in login_assignments
        ] if login_assignments else empty_choice

    return choices_dict


@require_http_methods(['GET'])
@responding_json
def get_employee_assignments(request, login):
    choices_dict = get_assignment_choices([login])
    return choices_dict[login]


def _get_phones_choices(logins):
    phones = (
        StaffPhone.objects
        .filter(
            staff__login__in=logins,
            kind=PHONE_KIND.COMMON,
            intranet_status=1,
        )
        .order_by('staff', 'position')
        .values_list('staff__login', 'number')
    )

    choices_dict = dict.fromkeys(logins, {'choices': []})

    for login, phone_no in phones:
        try:
            number = format_number(
                parse(phone_no),
                PhoneNumberFormat.INTERNATIONAL,
            )
        except NumberParseException:
            number = phone_no

        choices_dict[login]['choices'].append(
            {
                'label': number,
                'value': phone_no
            }
        )
    return choices_dict


@require_http_methods(['GET'])
@responding_json
def get_employee_phones(request, login):
    return _get_phones_choices([login])[login]


# ADMIN #


@require_permission('django_intranet_stuff.can_view_trip', exception=Http404)
@require_http_methods(['GET'])
def admin_trip_list(request):
    ITEMS_ON_PAGE = 12

    try:
        page = int(request.GET.get('page', '1'))
    except ValueError:
        page = 1

    filter_ = request.GET.get('filter', '')

    q = {}
    if filter_:
        try:
            q = loads(filter_)
        except Exception:
            filter_ += '<-- Error'

    trip_list = TripQuestionaryCollection().find(q=q)
    trip_list = trip_list.sort([('creation_time', -1)])
    paginator = Paginator(trip_list, ITEMS_ON_PAGE)

    try:
        trip_list = paginator.page(page)
    except (EmptyPage, InvalidPage):
        trip_list = paginator.page(paginator.num_pages)

    # Манкипатчим пагинатор, так как он не берет от гнератора лист,
    # преждечем получить len
    trip_list.object_list = list(trip_list.object_list)
    can_edit = request.user.has_perm('django_intranet_stuff.can_edit_trip')

    return render(
        request,
        'admin_trip_list.html',
        {
            'trip_list': trip_list,
            'filter_': filter_,
            'can_edit_trip': can_edit,
        }
    )


@require_permission('django_intranet_stuff.can_view_trip', exception=Http404)
@require_http_methods(['GET'])
@responding_json
def admin_trip(request, trip_uuid):
    trip = TripQuestionaryCollection().get(uuid=trip_uuid)

    data = trip.data
    del data['_id']

    return trip.data


@require_permission('django_intranet_stuff.can_edit_trip', exception=Http404)
@require_http_methods(['POST'])
def admin_trip_delete(request, trip_uuid):
    trip = TripQuestionaryCollection().get(uuid=trip_uuid)

    trip.delete()

    url = request.META.get('HTTP_REFERER', '')
    if not url:
        url = reverse('trip_api-admin-list')
    return HttpResponseRedirect(url)


@require_permission('django_intranet_stuff.can_edit_trip', exception=Http404)
@require_http_methods(['POST'])
def admin_trip_start_operators(request, trip_uuid):
    trip = TripQuestionaryCollection().get(uuid=trip_uuid)

    method = trip_async_operator
    if 'delay' in request.POST:
        method = method.delay
    method(trip_uuid=trip.uuid)

    kwargs = {'trip_uuid': trip.uuid}
    url = reverse('trip_api-admin-show', kwargs=kwargs)
    return HttpResponseRedirect(url)


@require_permission('django_intranet_stuff.can_view_trip', exception=Http404)
@require_http_methods(['GET'])
def admin_wierd_trip_list(request):
    ITEMS_ON_PAGE = 12

    try:
        page = int(request.GET.get('page', '1'))
    except ValueError:
        page = 1

    filter_ = request.GET.get('filter', '')

    trip_list = get_wierd_trips()
    trip_list = trip_list.sort([('creation_time', -1)])
    paginator = Paginator(trip_list, ITEMS_ON_PAGE)

    try:
        trip_list = paginator.page(page)
    except (EmptyPage, InvalidPage):
        trip_list = paginator.page(paginator.num_pages)

    # Манкипатчим пагинатор, так как он не берет от гнератора лист,
    # преждечем получить len
    trip_list.object_list = list(trip_list.object_list)

    can_edit = request.user.has_perm('django_intranet_stuff.can_edit_trip')

    return render(
        request,
        'admin_trip_list.html',
        {
            'trip_list': trip_list,
            'filter_': filter_,
            'can_edit_trip': can_edit,
        }
    )
