from collections import defaultdict
from typing import Any, Dict, List, Optional, Union

from staff.departments.models import Department
from staff.groups.models import Group
from staff.umbrellas.models import UmbrellaAssignment, Umbrella


def export_umbrella_assignments(persons: List[str]) -> Dict[str, List[Dict[str, Any]]]:
    assignments_query_set = UmbrellaAssignment.all_assignments.active().filter(person__login__in=persons)
    assignments = assignments_query_set.values(
        'person__login',
        'umbrella__issue_key',
        'engagement',
        'engaged_from',
        'engaged_to',
    )

    result: Dict[str, List[Dict[str, Any]]] = defaultdict(list)

    for assignment in assignments:
        engagement: Dict[str, Any] = {
            'goal': assignment['umbrella__issue_key'],
            'engagement': assignment['engagement'],
            'engaged_from': assignment['engaged_from'],
            'engaged_to': assignment['engaged_to'],
        }
        result[assignment['person__login']].append(engagement)

    return result


def export_umbrellas(
    continuation_token: Optional[int],
    batch_size: int,
) -> Dict[str, Union[int, List[Dict[str, Any]]]]:
    abc_service_id_mapping = dict(
        Group.objects
        .filter(url__in=Department.valuestreams.values_list('url', flat=True))
        .values_list('url', 'service_id')
    )

    umbrellas_qs = Umbrella.objects.active()

    if continuation_token is not None:
        umbrellas_qs = umbrellas_qs.filter(id__gt=continuation_token)

    umbrellas_qs = umbrellas_qs.order_by('pk')
    fetched_umbrellas: List[Dict[str, Any]] = []
    umbrellas = list(
        umbrellas_qs[:batch_size].values(
            'pk',
            'goal_id',
            'issue_key',
            'name',
            'value_stream__name',
            'value_stream__name_en',
            'value_stream__url',
        )
    )

    for umbrella in umbrellas:
        umbrella_data: Dict[str, Any] = {
            'goal': umbrella['issue_key'],
            'goal_id': umbrella['goal_id'],
            'name': umbrella['name'],
            'value_stream': {
                'name': umbrella['value_stream__name'],
                'name_en': umbrella['value_stream__name_en'],
                'url': umbrella['value_stream__url'],
                'abc_service_id': abc_service_id_mapping.get(umbrella['value_stream__url']),
            },
        }
        fetched_umbrellas.append(umbrella_data)

    result: Dict[str, Union[int, List[Dict[str, Any]]]] = {'result': fetched_umbrellas}

    if len(fetched_umbrellas) >= batch_size:
        result['continuation_token'] = umbrellas[-1]['pk']

    return result
