from datetime import datetime
from typing import Any, Dict, List

from staff.lib.db import atomic
from staff.person.models import Staff

from staff.umbrellas.models import UmbrellaAssignment


def update_umbrella_assignments(persons: List[Staff], umbrella_assignments: List[Dict[str, Any]]) -> None:
    """
    Batch operations avoided to male sure django signals are passed
    """
    now = datetime.now()
    new_engagements = {x['umbrella'] and x['umbrella'].id: x['engagement'] for x in umbrella_assignments}

    with atomic():
        current_assignments = (
            UmbrellaAssignment.all_assignments
            .active()
            .filter(person__in=persons)
            .select_for_update()
            .order_by('person_id')
        )
        new_person_umbrellas = {person.id: set(new_engagements.keys()) for person in persons}

        for current_assignment in current_assignments:
            new_engagement = new_engagements.get(current_assignment.umbrella_id)

            if current_assignment.engagement == new_engagement:
                new_person_umbrellas[current_assignment.person_id].discard(current_assignment.umbrella_id)
            else:
                current_assignment.engaged_to = now
                current_assignment.save()

        for person_id, umbrella_ids in new_person_umbrellas.items():
            for umbrella_id in umbrella_ids:
                new_assignment = UmbrellaAssignment.all_assignments.create(
                    person_id=person_id,
                    umbrella_id=umbrella_id,
                    engagement=new_engagements[umbrella_id],
                    engaged_from=now,
                    engaged_to=None,
                )
                new_assignment.save()
