from typing import Any, Dict, List

import sform

from django.core.exceptions import ValidationError

from staff.budget_position.models import BudgetPositionAssignment
from staff.departments.models import Department
from staff.lib.models.mptt import get_ancestors_query
from staff.person.models import Staff

from staff.umbrellas.models import Umbrella


class UmbrellaEngagementForm(sform.SForm):
    umbrella = sform.SuggestField(
        queryset=Umbrella.objects.active(),
        to_field_name='issue_key',
        label_fields=('name',),
        state=sform.NORMAL,
    )

    engagement = sform.DecimalField(
        min_value=0.0,
        max_value=100.0,
        decimal_places=3,
        state=sform.REQUIRED,
    )


class UmbrellaAssignmentForm(sform.SForm):
    TOTAL_ENGAGEMENT = 100

    persons = sform.GridField(
        sform.SuggestField(
            queryset=Staff.objects.active(),
            to_field_name='login',
            label_fields=('login',),
        ),
        state=sform.REQUIRED,
    )

    umbrellas = sform.GridField(
        sform.FieldsetField(UmbrellaEngagementForm),
        state=sform.REQUIRED,
    )

    @staticmethod
    def clean_persons(value: List[Staff]) -> List[Staff]:
        logins = {person.login for person in value}

        if len(logins) != len(value):
            raise ValidationError('Persons should be unique', 'persons_not_unique')

        return value

    def clean_umbrellas(self, value: List[Dict[str, Any]]) -> List[Dict[str, Any]]:
        issue_keys = {umbrella['umbrella'].issue_key if umbrella.get('umbrella') else None for umbrella in value}

        if len(issue_keys) != len(value):
            raise ValidationError('Umbrellas should be unique', 'umbrellas_not_unique')

        if sum(umbrella['engagement'] for umbrella in value) != self.TOTAL_ENGAGEMENT:
            raise ValidationError(f'Total engagement should be {self.TOTAL_ENGAGEMENT}', 'invalid_total_engagement')

        return value

    def clean(self):
        cleaned_data = self.cleaned_data

        if not self.is_valid():
            return cleaned_data

        persons = cleaned_data['persons']
        person_value_stream_ids: List[int] = list(
            BudgetPositionAssignment.objects
            .active()
            .filter(person__in=persons, main_assignment=True)
            .values_list('value_stream_id', flat=True)
        )

        if len(person_value_stream_ids) != len(persons):
            raise ValidationError('Persons should have main assignments', 'persons_without_main_assignment')

        person_value_stream_chains: Dict[int, List[int]] = self._get_value_stream_chains(person_value_stream_ids)

        for umbrella_engagement in cleaned_data['umbrellas']:
            umbrella = umbrella_engagement.get('umbrella')

            if umbrella:
                if any(umbrella.value_stream_id not in chain for chain in person_value_stream_chains.values()):
                    raise ValidationError(
                        'Umbrellas are from different value stream',
                        'umbrella_from_different_value_stream',
                    )

        return cleaned_data

    @staticmethod
    def _get_value_stream_chains(value_stream_ids: List[int]) -> Dict[int, List[int]]:
        value_stream_hierarchy_query = get_ancestors_query(
            Department.valuestreams.filter(id__in=value_stream_ids),
            include_self=True,
        )

        value_stream_parents_mapping = dict(
            Department.valuestreams.filter(value_stream_hierarchy_query).values_list('id', 'parent_id')
        )

        value_stream_chains: Dict[int, List[int]] = {}

        for value_stream_id in value_stream_ids:
            value_stream_chains[value_stream_id] = [value_stream_id]
            parent_id = value_stream_parents_mapping[value_stream_id]

            while parent_id is not None:
                value_stream_chains[value_stream_id].append(parent_id)
                parent_id = value_stream_parents_mapping[parent_id]

        return value_stream_chains
