from django.conf import settings
from pymongo.write_concern import WriteConcern

from staff.lib.mongodb import mongo


class PersonSettings(dict):

    MONGO_COLLECTION = 'person_settings'

    fields = {
        # имя, (default, prepare)
        'is_calendar_vertical': (True, bool),
    }

    def __init__(self, person_id):
        super(PersonSettings, self).__init__()
        self.collection = mongo.db.get_collection(
            self.MONGO_COLLECTION,
            write_concern=WriteConcern(settings.MONGO_WRITE_CONCERN_W),
        )
        self._person_id = person_id
        self.reload()

    def reload(self):
        self.clear()
        self.update(self._default())
        self.update(self._load())

        self.pop('person_id', None)
        self.pop('_id', None)

        return self

    def commit(self, data=None):
        data = data or {}

        _data = {}
        for name, (value, prepare) in self.fields.items():
            if name in data:
                value = prepare(data[name])
            elif name in self:
                value = self[name]
            _data[name] = value

        self._save(_data)
        return self.reload()

    def __setitem__(self, key, value):
        if key in self.fields:
            super(PersonSettings, self).__setitem__(key, self.fields[key][1](value))
        else:
            raise KeyError(key)

    def _load(self):
        return self.collection.find_one({'person_id': self._person_id}) or {}

    def _default(self):
        return {name: default for name, (default, _) in self.fields.items()}

    def _save(self, data):
        _data = {'person_id': self._person_id}
        _data.update(data)

        self.collection.replace_one({'person_id': self._person_id}, _data, upsert=True)
