import json

from rest_framework.response import Response
from rest_framework.views import APIView

from django.core.paginator import (
    Paginator,
    EmptyPage,
    PageNotAnInteger,
)
from django.core.exceptions import PermissionDenied
from django.http import Http404
from django.db.models import Q
from django.views.decorators.http import require_http_methods
from django.shortcuts import render

from staff.person.models import Staff

from staff.lib.decorators import responding_json
from staff.lib.utils.paging import page_range
from .domain import get_user_settings
from .serializers import UserSettingsSerializer
from .objects import PersonSettings
from staff.lib.decorators import available_for_external


@available_for_external('achievery.external_with_achievery_access')
class UserSettingsAPIView(APIView):
    serializer_class = UserSettingsSerializer

    def get_person(self):
        return self.request.user.get_profile()

    def get_settings(self):
        person = self.get_person()
        return get_user_settings(person)

    def get(self, request):
        settings = self.get_settings()
        serializer = self.serializer_class(settings)
        return Response(serializer.data)

    def post(self, request):
        settings = self.get_settings()
        serializer = self.serializer_class(settings, data=request.DATA)

        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data)
        else:
            return Response(serializer.errors)


@require_http_methods(['GET'])
def persons_list(request):
    if not request.user.is_superuser:
        raise PermissionDenied()

    query = request.GET.get('query', '').strip()

    if query:
        try:
            filter_by = {'id': int(query)}
        except ValueError:
            filter_by = {
                'first_name__contains': query,
                'last_name__contains': query,
                'login__contains': query,
            }
    else:
        filter_by = {}

    paginator = Paginator(_persons_list(**filter_by), 20)
    try:
        persons = paginator.page(request.GET.get('page'))
    except PageNotAnInteger:
        persons = paginator.page(1)
    except EmptyPage:
        persons = paginator.page(paginator.num_pages)

    context = {
        'persons': persons,
        'pages_nums': page_range(persons.number, paginator.num_pages, 10),
        'query': query,
    }

    return render(request, 'persons.html', context)


def _persons_list(**filter_by):
    qs = (
        Staff.objects
        .values('id', 'login', 'first_name', 'last_name')
        .order_by('login')
    )

    if filter_by:
        if 'id' in filter_by:
            qs = qs.filter(id=filter_by['id'])
        else:
            or_q = Q()
            for field, value in filter_by.items():
                or_q |= Q(**{field: value})
            qs = qs.filter(or_q)

    return qs


@require_http_methods(['GET', 'POST'])
@responding_json
def person_settings(request, login):
    if not request.user.is_superuser:
        raise PermissionDenied()

    try:
        person_id = Staff.objects.values_list('id', flat=True).get(login=login)
    except Staff.DoesNotExist:
        raise Http404

    if request.method == 'POST':
        return PersonSettings(person_id).commit(json.loads(request.POST['json']))

    return PersonSettings(person_id)
