from itertools import chain
from typing import Any, Dict, List

from staff.departments.models import DepartmentStaff, DepartmentRoles
from staff.person.models import Staff
from staff.lib.models.departments_chain import get_departments_tree

from staff.verification.models import ExternalLogin


def export(internal_logins: List[str], internal_uids: List[str]) -> Dict[str, Dict[str, Any]]:
    fields = ('id', 'uid', 'login', 'is_dismissed', 'is_homeworker', 'affiliation', 'department_id')
    persons_qs = Staff.objects.filter(is_robot=False).values(*fields)

    if internal_logins:
        persons_qs = persons_qs.filter(login__in=internal_logins)

    if internal_uids:
        persons_qs = persons_qs.filter(uid__in=internal_uids)

    department_id_by_uid = {p['uid']: p['department_id'] for p in persons_qs}
    department_chains = get_departments_tree(list(department_id_by_uid.values()), fields=['id'])

    all_department_ids = set(
        [ch['id'] for ch in chain.from_iterable(department_chains.values())]
    )

    chief_uid_by_department_id = dict(
        DepartmentStaff.objects
        .filter(department_id__in=all_department_ids, role_id=DepartmentRoles.CHIEF.value)
        .values_list('department_id', 'staff__uid')
    )

    chief_by_uid = {}
    for uid, department_id in department_id_by_uid.items():
        dep_chain = reversed(department_chains[department_id])
        chief_chain = [chief_uid_by_department_id.get(dep_id['id']) for dep_id in dep_chain]
        chief_chain = [
            chief_uid
            for chief_uid in chief_chain
            if chief_uid and chief_uid != uid
        ]
        chief_by_uid[uid] = chief_chain and chief_chain[0] or None

    persons = {
        p['login']: {
            'internal_login': p['login'],
            'internal_uid': p['uid'],
            'is_dismissed': p['is_dismissed'],
            'is_homeworker': p['is_homeworker'],
            'affiliation': p['affiliation'],
            'external_login': None,
            'external_uid': None,
        } for p in persons_qs  # здесь результат запроса уже закеширован
    }

    external_logins_qs = (
        ExternalLogin.objects
        .filter(status_active=True, person__login__in=persons)
        .values('person__login', 'login', 'uid')
        .order_by('created_at')
    )

    for ext_login_data in external_logins_qs:
        login = ext_login_data.pop('person__login')
        persons[login]['external_login'] = ext_login_data['login']
        persons[login]['external_uid'] = ext_login_data['uid']

    for person_id, person_data in persons.items():
        person_data['chief_uid'] = chief_by_uid[person_data['internal_uid']]

    return persons
