from operator import attrgetter

import yenv
from django.db.models import Q
from typing import List

from staff.person.effects import actualize_yandex_disk
from staff.person.controllers import PersonCtl
from staff.person.dis_staff_services import StaffService

from .tasks import sync_external_logins
from .models import ExternalLogin


class ExternalLoginValidationError(Exception):
    pass


class OAuthException(Exception):
    pass


def is_chief(first_user, second_user):
    """Проверка руководитель (chief) - подчиненный

    Проверяем является ли first_user руководителем в цепочке
    департаментов сотрудника second_user
    """
    return first_user in StaffService(second_user).get_chiefs()


def can_manage_ext_login(target, confirmer):
    return (
        target == confirmer or
        (
            confirmer.user.is_superuser or (
                confirmer.user.has_perm(
                    'django_intranet_stuff.can_manage_external_logins'
                )
                and
                is_chief(confirmer, target)
            )
        )
    )


class ExternalLoginCtl(object):

    def __init__(self, target, observer=None):
        self.target = target
        self.observer = observer
        self._persons_to_update = {}

    def activate(self, ext_uid, ext_login):
        logins = self._get_logins(ext_uid)
        self._deactivate(logins, ext_uid)
        self._activate(logins, ext_uid, ext_login)
        updated_persons_controllers = self._updated_persons_controllers()
        self._save_persons(updated_persons_controllers)
        actualize_yandex_disk(self.target)

        self._sync_external_logins(updated_persons_controllers)

    def deactivate(self):
        logins = self._get_logins()
        self._deactivate(logins)
        updated_persons_controllers = self._updated_persons_controllers()
        self._save_persons(updated_persons_controllers)
        actualize_yandex_disk(self.target)

        self._sync_external_logins(updated_persons_controllers)

    def _sync_external_logins(self, updated_persons_controllers):
        task_kwargs = {
            'person_ids': [p.id for p in updated_persons_controllers]
        }

        if yenv.type != 'development':
            sync_external_logins.apply_async(kwargs=task_kwargs, countdown=30)
        else:
            sync_external_logins(**task_kwargs)

    def _get_logins(self, new_ext_uid=None) -> List[ExternalLogin]:
        qs = ExternalLogin.objects.select_related('person').select_for_update()

        if new_ext_uid is None:
            qs = qs.filter(person=self.target)
        else:
            qs = qs.filter(
                Q(uid=new_ext_uid, status_active=True) | Q(person=self.target)
            )

        return list(qs)

    def _update_person(self, person, new_ext_login):
        try:
            person_ctl = self._persons_to_update[person.id]
        except KeyError:
            person_ctl = PersonCtl(person)
            self._persons_to_update[person.id] = person_ctl

        person_ctl.login_passport = new_ext_login
        person_ctl.is_login_passport_confirmed = bool(new_ext_login)

    def _updated_persons_controllers(self):
        # Порядок важен из-за unique на login_passport
        controllers = sorted(
            self._persons_to_update.values(),
            key=attrgetter('is_login_passport_confirmed'),
        )

        return controllers

    def _save_persons(self, updated_persons_controllers):
        for person_ctl in updated_persons_controllers:
            person_ctl.save(self.observer)

        self._persons_to_update = {}

    def _update_login(self, login, is_active):
        login.status_active = is_active
        login.ext_passport_synced = False
        login.save()

    def _deactivate(self, logins: List[ExternalLogin], new_ext_uid=None):
        for login in logins:
            if login.status_active and (login.person != self.target or login.uid != new_ext_uid):
                self._update_login(login, False)
                self._update_person(login.person, None)

    def _activate(self, ext_logins: List[ExternalLogin], new_ext_uid, new_ext_login):
        try:
            login = next(
                ext_login
                for ext_login in ext_logins
                if ext_login.uid == new_ext_uid and ext_login.person == self.target
            )
        except StopIteration:
            login = ExternalLogin(person=self.target, login=new_ext_login, uid=new_ext_uid)

        self._update_login(login, True)
        self._update_person(login.person, new_ext_login)
