import logging
from time import sleep

from staff.celery_app import app

from staff.lib.db import atomic
from staff.person.passport.external import sync_with_external_passport

from staff.verification.models import ExternalLogin


logger = logging.getLogger('staff.sync_with_external_passport')


@app.task()
def sync_external_logins(person_ids=None, force=False):
    person_ids = _get_person_ids(person_ids, force)

    for person_id, is_active in person_ids:
        try:
            _sync_state(person_id=person_id, is_active=is_active, force=force)
        except Exception:
            logger.exception(
                'Exception when sync "%s" logins for person with id=%s, force=%s',
                'active' if is_active else 'unactive', person_id, force
            )


def _get_person_ids(person_ids, force):
    person_ids_qs = (
        ExternalLogin.objects
        .values_list('person_id', 'status_active')
        .order_by('status_active')
        .distinct()
    )

    if not force:
        person_ids_qs = _force_filter(person_ids_qs)

    if person_ids is not None:
        person_ids_qs = person_ids_qs.filter(person_id__in=person_ids)

    return person_ids_qs


def _force_filter(qs):
    return qs.filter(
        ext_passport_synced=False
    )


def _sync_state(person_id, is_active, force):
    with atomic():
        ext_logins = (
            ExternalLogin.objects
            .filter(person_id=person_id, status_active=is_active)
            .select_related('person')
            .select_for_update()
        )

        if not force:
            ext_logins = _force_filter(ext_logins)

        for login in ext_logins:
            changed = False
            if force or not login.ext_passport_synced:
                try:
                    sleep(1)  # Чтобы не укладывать паспорт
                    sync_with_external_passport(login)
                except Exception:
                    logger.exception(
                        'Exception during sync with external passport for login %s',
                        login.person.login
                    )
                else:
                    login.ext_passport_synced = True
                    changed = True

            if changed:
                login.save()
