from itertools import chain
import logging
from typing import Any, Dict, Optional, Tuple

import pytz
from IPy import IPint

from django.conf import settings

from staff.map.models import Office
from staff.person.models import Staff

from staff.whistlah.models import OfficeNet, StaffLastOffice


logger = logging.getLogger(name='staff.whistlah.utils')


# Типы записей
NET_FIRST_IP, WHISTLAH_RECORD, NET_LAST_IP = list(range(-1, 2))

# Индексы в списке
I__IP_VAL, I__TYPE, I__OBJ = list(range(3))

local_tz = pytz.timezone(settings.TIME_ZONE)


def raw_record_to_result(
    updated_at,
    office_id: Optional[int],
    office_name_pair: Optional[Tuple[Optional[str], Optional[str]]],
) -> Dict[str, Any]:
    office_id = _fix_legacy_fake_vpn_office_network(office_id)
    fixed_office_name_pair = (office_name_pair or ('VPN', 'VPN'))

    return {
        'updated_at': updated_at,
        'office_id': office_id,
        'name': fixed_office_name_pair[0],
        'name_en': fixed_office_name_pair[1],
        'is_vpn': office_id is None,
    }


def _fix_legacy_fake_vpn_office_network(office_id: Optional[int]) -> Optional[int]:
    if office_id == 0:
        office_id = None

    return office_id


def parse_postgres_activity(raw_data) -> Dict[str, Dict[str, Any]]:
    if not raw_data:
        return {}

    unique_hosts = {host for updated_at, host, login in raw_data}
    ip_to_office_ids = get_offices(unique_hosts)
    office_names = get_office_names(ip_to_office_ids.values())

    result: Dict[str, Dict[str, Any]] = {}

    for updated_at, host, login in raw_data:
        office_id = ip_to_office_ids[host]
        office_name_pair = office_names.get(office_id)
        result_record = raw_record_to_result(updated_at, office_id, office_name_pair)

        if login not in result or result[login]['updated_at'] < result_record['updated_at']:
            result[login] = result_record

    return result


def _get_networks():
    yield from OfficeNet.objects.values_list('office_id', 'first_ip_value', 'last_ip_value')


def _get_networks_bounds():
    for office_id, first_ip_value, last_ip_value in _get_networks():
        yield [first_ip_value, NET_FIRST_IP, office_id or 0]
        yield [last_ip_value, NET_LAST_IP, office_id or 0]


def _get_ips_records(ips):
    for ip in ips:
        if ip:
            yield [IPint(ip).int(), WHISTLAH_RECORD, ip]


def get_offices(ips):
    """
    Возвращаем dict вида IP: office_id
    office_id == 0 если IP относится к подсети VPN
    office_id == None если IP относится к неизвестной подсети
    """
    ips = sorted(chain(_get_networks_bounds(), _get_ips_records(ips)))
    office_net = None
    office_ids = {}
    for ip in ips:
        if ip[I__TYPE] == NET_FIRST_IP:
            office_net = ip[I__OBJ]
        elif ip[I__TYPE] == NET_LAST_IP:
            office_net = None
        elif ip[I__TYPE] == WHISTLAH_RECORD:
            office_ids[ip[I__OBJ]] = office_net
    return office_ids


def get_office_names(ids):
    """
    Возвращаем дикт вида id: (name, name_en)
    для всех офисов, id которых переданы в параметре
    """
    office_names = Office.objects.filter(id__in=set(ids)).values_list(
        'id', 'name', 'name_en')
    return {o[0]: (o[1], o[2]) for o in office_names}


def get_saved_activities(logins):
    activities = (
        StaffLastOffice.objects
        .filter(staff__login__in=logins)
        .values(
            'staff__login',
            'office_id',
            'office_name',
            'office_name_en',
            'is_vpn',
            'updated_at',
            'office__office_parts__part_of_main_office__code',
            'office__code',
        )
        .order_by('staff__login', 'office_id')
    )

    result = {}
    for activity in activities:
        office_code = (
            activity['office__office_parts__part_of_main_office__code']
            if activity['office__office_parts__part_of_main_office__code'] is not None
            else activity['office__code']
        )

        if activity['staff__login'] in result:
            result[activity['staff__login']]['office_parts_codes'].append(office_code)
        else:
            result[activity['staff__login']] = {
                'is_vpn': activity['is_vpn'],
                'office_id': activity['office_id'],
                'updated_at': activity['updated_at'],
                'name': activity['office_name'],
                'name_en': activity['office_name_en'],
                'office_parts_codes': [office_code],
            }

    return result


def get_last_activity(logins):
    logins_set = set(logins)
    result = {}

    if logins_set:
        result.update(get_saved_activities(logins_set))
        logins_set -= set(result.keys())

    if logins_set:
        for login in logins_set:
            result[login] = None

    return result


def round_parse_result_to_minute(last_activities: Dict[str, Dict[str, Any]]) -> None:
    for last_activity in last_activities.values():
        last_activity['updated_at'] = last_activity['updated_at'].replace(second=0, microsecond=0)


def replace_login_with_id(last_activities: Dict[str, Dict[str, Any]]) -> Dict[int, Dict[str, Any]]:
    staffs = dict(
        Staff.objects
        .filter(login__in=last_activities.keys())
        .values_list('login', 'id')
    )

    result = {}
    for login, last_activity in last_activities.items():
        staff_id = staffs.get(login)
        if staff_id:
            last_activity['staff_id'] = staff_id
            result[staff_id] = last_activity

    return result


def remove_already_existing_activities(last_activities: Dict[int, Dict[str, Any]]) -> Dict[int, Dict[str, Any]]:
    existing_records = dict(
        StaffLastOffice.objects
        .filter(staff_id__in=last_activities.keys())
        .values_list('staff_id', 'updated_at')
    )

    result = {}
    for staff_id, last_activity in last_activities.items():
        updated_at = existing_records.get(staff_id)
        if not updated_at or updated_at < last_activity['updated_at']:
            result[staff_id] = last_activity

    return result
