from collections import defaultdict
from json import dumps
from functools import partial

from django.contrib.auth.decorators import permission_required
from django.db.models import FieldDoesNotExist
from django.http import HttpResponse
from django.views.decorators.http import require_http_methods

from staff.map.models import Office, City

from staff.lib.decorators import responding_json
from staff.lib.utils import qs_values
from staff.whistlah.models import NOCOffice

from staff.whistlah.utils import get_offices
from staff.whistlah.models import StaffLastOffice


def serialize_model(obj, relations=None, fields=None):
    if not relations:
        relations = {}

    if not fields:
        fields = [field.name for field in obj.__class__._meta.fields]

    def field_value(field):
        value = getattr(obj, field.name)

        if field.rel and value and field.name in relations:
            if type(relations[field.name]) == dict and '_fields' in relations[field.name]:
                fields = relations[field.name].get('_fields', None)
            else:
                fields = None

            return serialize_model(value, relations[field.name], fields)

        return str(value) if value is not None else ''

    def generate():
        for field in fields:
            try:
                field = obj.__class__._meta.get_field(field)
            except FieldDoesNotExist:
                if hasattr(obj, field):
                    yield field, getattr(obj, field)
            else:
                yield field.name, field_value(field)

    return dict(generate())


VPN_STUB = {
    'code': '',
    'intranet_status': '1',
    'from_staff_id': '',
    'id': '',
    'city': {
        "intranet_status": '1',
        "color": '000000',
        "country": '',
        "created_at": '',
        "modified_at": '',
        "geo_id": '0',
        "position": '0',
        "name_en": '',
        "native_lang": '',
        "id": '',
        "name": 'VPN'
    },
    'tz': '',
    'is_virtual': 'False',
    'address2_en': '',
    'color': '',
    'filter_id': '',
    'native_lang': '',
    'fax': '',
    'address1': '',
    'address2': '',
    'have_map': 'False',
    'name_en': '',
    'name': 'VPN',
    'coord_y': '',
    'coord_x': '',
    'phone': '',
    'created_at': '',
    'modified_at': '',
    'zoom': '0',
    'address1_en': '',
    'position': '0',
}


def office_by_ip(request, ip):
    try:
        office_id = get_offices([ip])[ip]
        if office_id:
            office = Office.objects.select_related('city').get(id=office_id)
        elif office_id == 0:
            return HttpResponse(dumps(VPN_STUB), content_type='text/javascript')
        else:
            office = {}

        if office:
            office.have_map = bool(office.have_map)  # STAFF-4611
            office = serialize_model(office, {'city': {}})

        return HttpResponse(dumps(office), content_type='text/javascript')
    except ValueError:
        return HttpResponse('Invalid ip', status=404)


class Where(object):
    def __call__(self, request, login):
        office = {}
        try:
            where = StaffLastOffice.objects.select_related('office', 'office__city').get(staff__login_ld=login)
            if where.office:
                office = serialize_model(where, {'office': {'city': {}}})
                del office['staff']
            elif where.is_vpn:
                office = serialize_model(where, {})
                office['office'] = serialize_model(Office(name='VPN'), {})
                office['office']['city'] = serialize_model(City(name='VPN'), {})
                del office['staff']
        except StaffLastOffice.DoesNotExist:
            pass
        return HttpResponse(dumps(office), content_type='text/javascript')


def where(request, login):
    return Where()(request, login)


@permission_required('whistlah.can_export_nocoffice_mapping')
@require_http_methods(['GET'])
@responding_json
def racktables_office_ids(request):
    offices_mapping = (
        NOCOffice.objects
        .values_list('office_id', 'noc_office_id')
        .order_by('office_id', 'noc_office_id')
    )
    result = defaultdict(list)
    for office_id, noc_office_id in offices_mapping:
        result[office_id].append(noc_office_id)

    return dict(result)


@responding_json
def where_bulk(request):
    last_activity_fields = (
        'is_vpn',
        'updated_at',
        'office__id',
        'office__name',
        'office__name_en',
        'office__city__id',
        'office__city__name',
        'office__city__name_en',
        'staff__id',
        'staff__login',
        'staff__first_name',
        'staff__first_name_en',
        'staff__last_name',
        'staff__last_name_en',
    )
    logins = request.GET.get('logins', '').split(',')
    objects = StaffLastOffice.objects.filter(staff__login__in=logins)
    serialized = objects.values(*last_activity_fields)
    normalizer = partial(
        qs_values.normalize_serialized,
        replacements={'staff': 'person'},
        datetime_fields=('updated_at',),
    )
    return list(map(normalizer, serialized))


_where_am_i_tpl = """<?xml version="1.0" encoding="UTF-8"?>
<code%(vpn)s>%(code)s</code>"""


def where_am_i_for_streaming(request):
    if 'HTTP_X_REAL_IP' in request.META:
        ip = request.META['HTTP_X_REAL_IP']
    elif 'HTTP_X_FORWARDED_FOR' in request.META:
        ip = request.META['HTTP_X_FORWARDED_FOR']
    else:
        ip = request.META['REMOTE_ADDR']
    office_id = get_offices([ip])[ip]
    data = {'vpn': '', 'code': ''}
    if office_id:
        data['code'] = (Office.objects.filter(id=office_id)
                        .values_list('code', flat=True)[0])
    elif office_id == 0:
        data['vpn'] = ' is_vpn="true"'
    return HttpResponse(_where_am_i_tpl % data, content_type='text/xml')
