from decimal import Decimal
from typing import Dict, List, Union

import sform

from staff.map.models import Room
from staff.workspace_management.models import BusinessUnit


class ShareForm(sform.SForm):
    business_unit = sform.SuggestField(
        queryset=BusinessUnit.objects.active(),
        label_fields=('id', ),
        state=sform.REQUIRED,
    )
    share_value = sform.DecimalField(
        min_value=Decimal(0.001),
        max_value=Decimal(100),
        decimal_places=4,
        state=sform.REQUIRED,
    )


class RoomSharePieForm(sform.SForm):
    room = sform.SuggestField(
        queryset=Room.objects.active(),
        label_fields=('id',),
        state=sform.REQUIRED,
    )
    # The Pentagon has the largest ground area. Each exterior wall measures 281m (921ft).
    # Its perimeter is 1,405m (4,610ft). It has five storeys with a total floor area of 604,000m2
    room_area = sform.DecimalField(
        min_value=Decimal(0.0),
        max_value=Decimal(604000),
        decimal_places=4,
        state=sform.REQUIRED,
    )
    shares = sform.GridField(sform.FieldsetField(ShareForm))

    def clean_shares(self, value) -> List[Dict[str, Union[Decimal, BusinessUnit]]]:
        shares_sum = sum(share['share_value'] for share in value)
        if shares_sum > Decimal(100):
            raise sform.ValidationError('Sum of shared should not be greater than 100', code='invalid_value')

        return value
