import uuid

from django.db import models

from staff.lib.models.base import TrackableModel
from staff.map.models import Office, Room
from staff.person.models import Organization, Staff


class BusinessUnit(TrackableModel):
    organization = models.OneToOneField(Organization, limit_choices_to={'intranet_status': 1})
    name = models.CharField(max_length=50)
    name_en = models.CharField(max_length=50, default='')

    def __str__(self):
        return f'{self.id}.{self.organization_id} {self.name}'


class RoomSharePie(models.Model):
    id = models.UUIDField(primary_key=True, null=False, default=uuid.uuid1)
    room = models.ForeignKey(Room, related_name='pies', null=False)
    created_at = models.DateTimeField(auto_now_add=True)
    author = models.ForeignKey(Staff, null=False)
    from_date = models.DateField(null=False)
    to_date = models.DateField(null=False)
    room_area = models.DecimalField(null=False, max_digits=12, decimal_places=4)

    def __str__(self):
        return f'Pie {self.id} in {self.room_id} ({self.room_area}sqm) ({self.from_date} - {self.to_date})'


class Share(models.Model):
    room_share_pie = models.ForeignKey(RoomSharePie, related_name='shares', null=False, on_delete=models.CASCADE)
    business_unit = models.ForeignKey(BusinessUnit, related_name='+', null=False)
    share_value = models.DecimalField(null=False, max_digits=7, decimal_places=4)

    def __str__(self):
        return f'Share {self.id} in {self.room_share_pie_id} of BU {self.business_unit_id} - {self.share_value}%'


class OfficeArea(models.Model):
    id = models.UUIDField(primary_key=True, null=False, default=uuid.uuid1)
    office = models.ForeignKey(Office, related_name='areas', null=False)
    created_at = models.DateTimeField(auto_now_add=True)
    from_date = models.DateField(null=False)
    to_date = models.DateField(null=False)
    office_area = models.DecimalField(null=False, max_digits=12, decimal_places=4)

    def __str__(self):
        return f'OfficeArea({self.id}) of {self.office_id} is {self.office_area}m2'


class Permissions(models.Model):
    class Meta:
        managed = False  # No database table creation or deletion operations will be performed for this model
        permissions = (
            ('can_manage_workspace_areas', 'Can manage workspace areas'),
        )
