from decimal import Decimal
from itertools import groupby
from typing import Dict, List

import attr

from staff.workspace_management.usage_report.office_area import OfficeAreaData
from staff.workspace_management.usage_report.shares_qs import shares_qs


@attr.s(auto_attribs=True)
class BCUsageModelRow:
    office_id: int
    office_name: str
    office_name_en: str

    bc_rent_area: Decimal
    work_area: Decimal
    public_area: Decimal
    bu_work_area: Decimal
    free_area: Decimal


def bc_usage_model(office_areas: Dict[int, OfficeAreaData]) -> List[BCUsageModelRow]:
    shares = groupby(shares_qs(list(office_areas.keys())), lambda row: row['room_share_pie__room__floor__office_id'])
    used_areas: Dict[int, Decimal] = {
        office_id: Decimal(sum(row['used_area'] for row in rows))
        for office_id, rows in shares
    }

    result = []
    for office_id, office_area in office_areas.items():
        used_area = used_areas[office_id] if office_id in used_areas else Decimal(0)
        result.append(BCUsageModelRow(
            office_id=office_id,
            office_name=office_area.office_name,
            office_name_en=office_area.office_name_en,
            bc_rent_area=office_area.office_area,
            work_area=office_area.rooms_area,
            public_area=office_area.public_area,
            bu_work_area=used_area,
            free_area=office_area.rooms_area - used_area,
        ))

    return result
