from decimal import Decimal
from itertools import groupby
from typing import Dict, List, Set, Tuple

import attr

from staff.workspace_management.models import BusinessUnit
from staff.workspace_management.usage_report.office_area import OfficeAreaData
from staff.workspace_management.usage_report.shares_qs import shares_qs


@attr.s(auto_attribs=True)
class BUUsageModelRow:
    office_id: int
    office_name: str
    office_name_en: str
    bu_name: str
    bu_name_en: str

    bu_working_area: Decimal
    bu_working_area_share: Decimal
    bu_public_area: Decimal
    bu_total_area: Decimal


def _bu_names(bu_ids: Set[int]) -> Dict[int, Tuple[str, str]]:
    return {
        bu_id: (name, name_en)
        for bu_id, name, name_en in BusinessUnit.objects.filter(id__in=bu_ids).values_list('id', 'name', 'name_en')
    }


def bu_usage_model(office_areas: Dict[int, OfficeAreaData]) -> List[BUUsageModelRow]:
    shares_list = list(shares_qs(list(office_areas.keys())))
    bu_names = _bu_names({share['business_unit_id'] for share in shares_list})
    shares = groupby(shares_list, lambda row: row['room_share_pie__room__floor__office_id'])

    result = []
    for office_id, bu_rows in shares:
        for bu_row in bu_rows:
            bu_id = bu_row['business_unit_id']
            bu_share = bu_row['used_area'] / office_areas[office_id].rooms_area
            bu_public_area = bu_share * office_areas[office_id].public_area
            result.append(BUUsageModelRow(
                office_id=office_id,
                office_name=office_areas[office_id].office_name,
                office_name_en=office_areas[office_id].office_name_en,
                bu_name=bu_names[bu_id][0],
                bu_name_en=bu_names[bu_id][1],
                bu_working_area=bu_row['used_area'],
                bu_working_area_share=bu_share * 100,
                bu_public_area=bu_public_area,
                bu_total_area=bu_row['used_area'] + bu_public_area,
            ))

    return result
